import { useState } from "react";
import { 
  Sheet,
  SheetContent,
  SheetDescription,
  SheetHeader,
  SheetTitle,
  SheetTrigger,
} from "@/components/ui/sheet";
import { Button } from "@/components/ui/button";
import { Separator } from "@/components/ui/separator";
import { Slider } from "@/components/ui/slider";
import { Switch } from "@/components/ui/switch";
import { Label } from "@/components/ui/label";
import { 
  Layers, 
  Zap, 
  Car, 
  Eye, 
  EyeOff,
  Filter,
  Battery,
  MapPin
} from "lucide-react";
import { cn } from "@/lib/utils";

interface FilterState {
  showRegions: boolean;
  showV2GCalls: boolean;
  showGridInvitations: boolean;
  showVehicles: boolean;
  showScoreBubbles: boolean;
  showHotZones: boolean;
  
  // Vehicle filters
  showCharging: boolean;
  showDischarging: boolean;
  showIdle: boolean;
  showOffline: boolean;
  showSocInfo: boolean;
  
  // Advanced filters
  socRange: [number, number];
  batteryRange: [number, number];
}

interface FiltersSheetProps {
  filters: FilterState;
  onFiltersChange: (filters: Partial<FilterState>) => void;
  vehicleCount?: number;
  trigger?: React.ReactNode;
}

export function FiltersSheet({ 
  filters, 
  onFiltersChange, 
  vehicleCount = 0,
  trigger 
}: FiltersSheetProps) {
  const [open, setOpen] = useState(false);

  const layerControls = [
    {
      key: "showRegions" as keyof FilterState,
      label: "Regions",
      icon: Layers,
      description: "Regional aggregated data"
    },
    {
      key: "showV2GCalls" as keyof FilterState,
      label: "V2G Calls",
      icon: Zap,
      description: "Grid demand signals"
    },
    {
      key: "showGridInvitations" as keyof FilterState,
      label: "Grid Invites",
      icon: MapPin,
      description: "Connection opportunities"
    },
    {
      key: "showVehicles" as keyof FilterState,
      label: "Vehicles",
      icon: Car,
      description: "Individual vehicle markers"
    },
    {
      key: "showScoreBubbles" as keyof FilterState,
      label: "Score Bubbles",
      icon: Eye,
      description: "Vehicle eligibility scores"
    },
    {
      key: "showHotZones" as keyof FilterState,
      label: "Hot Zones",
      icon: Filter,
      description: "High-priority grid areas"
    }
  ];

  const vehicleFilters = [
    {
      key: "showCharging" as keyof FilterState,
      label: "Charging",
      icon: Battery,
      color: "text-green-500"
    },
    {
      key: "showDischarging" as keyof FilterState,
      label: "Discharging", 
      icon: Zap,
      color: "text-blue-500"
    },
    {
      key: "showIdle" as keyof FilterState,
      label: "Idle",
      icon: Car,
      color: "text-yellow-500"
    },
    {
      key: "showOffline" as keyof FilterState,
      label: "Offline",
      icon: EyeOff,
      color: "text-red-500"
    },
    {
      key: "showSocInfo" as keyof FilterState,
      label: "SOC Info",
      icon: Battery,
      color: "text-purple-500"
    }
  ];

  const handleFilterChange = (key: keyof FilterState, value: boolean | [number, number]) => {
    onFiltersChange({ [key]: value });
  };

  const defaultTrigger = (
    <Button 
      variant="outline" 
      size="sm"
      className="gap-2"
      data-testid="button-open-filters"
    >
      <Filter className="h-4 w-4" />
      Filters
    </Button>
  );

  return (
    <Sheet open={open} onOpenChange={setOpen}>
      <SheetTrigger asChild>
        {trigger || defaultTrigger}
      </SheetTrigger>
      
      <SheetContent 
        side="bottom" 
        className="h-[80vh] overflow-y-auto"
        data-testid="filters-sheet"
      >
        <SheetHeader className="text-left">
          <SheetTitle>Map Filters & Layers</SheetTitle>
          <SheetDescription>
            Customize what you see on the V2G platform map
          </SheetDescription>
        </SheetHeader>

        <div className="space-y-6 py-4">
          {/* Layer Controls */}
          <div className="space-y-4">
            <h3 className="font-medium text-sm">Map Layers</h3>
            <div className="grid gap-3">
              {layerControls.map((control) => {
                const Icon = control.icon;
                const isActive = filters[control.key] as boolean;
                
                return (
                  <div 
                    key={control.key}
                    className="flex items-center justify-between p-3 rounded-lg border bg-card/50"
                  >
                    <div className="flex items-center gap-3">
                      <Icon className={cn(
                        "h-4 w-4",
                        isActive ? "text-primary" : "text-muted-foreground"
                      )} />
                      <div>
                        <div className="font-medium text-sm">{control.label}</div>
                        <div className="text-xs text-muted-foreground">
                          {control.description}
                        </div>
                      </div>
                    </div>
                    <Switch
                      checked={isActive}
                      onCheckedChange={(checked) => handleFilterChange(control.key, checked)}
                      data-testid={`switch-${control.key}`}
                    />
                  </div>
                );
              })}
            </div>
          </div>

          <Separator />

          {/* Vehicle Status Filters */}
          <div className="space-y-4">
            <div className="flex items-center justify-between">
              <h3 className="font-medium text-sm">Vehicle Filters</h3>
              <span className="text-xs text-muted-foreground">
                {vehicleCount} vehicles
              </span>
            </div>
            <div className="grid gap-3">
              {vehicleFilters.map((filter) => {
                const Icon = filter.icon;
                const isActive = filters[filter.key] as boolean;
                
                return (
                  <div 
                    key={filter.key}
                    className="flex items-center justify-between p-3 rounded-lg border bg-card/50"
                  >
                    <div className="flex items-center gap-3">
                      <Icon className={cn("h-4 w-4", filter.color)} />
                      <span className="font-medium text-sm">{filter.label}</span>
                    </div>
                    <Switch
                      checked={isActive}
                      onCheckedChange={(checked) => handleFilterChange(filter.key, checked)}
                      data-testid={`switch-${filter.key}`}
                    />
                  </div>
                );
              })}
            </div>
          </div>

          <Separator />

          {/* Advanced Filters */}
          <div className="space-y-4">
            <h3 className="font-medium text-sm">Advanced Filters</h3>
            
            {/* SOC Range */}
            <div className="space-y-2">
              <Label className="text-sm">State of Charge (%)</Label>
              <div className="px-2">
                <Slider
                  value={filters.socRange}
                  onValueChange={(value) => handleFilterChange('socRange', value as [number, number])}
                  max={100}
                  min={0}
                  step={5}
                  className="w-full"
                  data-testid="slider-soc-range"
                />
                <div className="flex justify-between text-xs text-muted-foreground mt-1">
                  <span>{filters.socRange[0]}%</span>
                  <span>{filters.socRange[1]}%</span>
                </div>
              </div>
            </div>

            {/* Battery Capacity Range */}
            <div className="space-y-2">
              <Label className="text-sm">Battery Capacity (kWh)</Label>
              <div className="px-2">
                <Slider
                  value={filters.batteryRange}
                  onValueChange={(value) => handleFilterChange('batteryRange', value as [number, number])}
                  max={150}
                  min={10}
                  step={5}
                  className="w-full"
                  data-testid="slider-battery-range"
                />
                <div className="flex justify-between text-xs text-muted-foreground mt-1">
                  <span>{filters.batteryRange[0]} kWh</span>
                  <span>{filters.batteryRange[1]} kWh</span>
                </div>
              </div>
            </div>
          </div>

          {/* Quick Actions */}
          <div className="flex gap-2 pt-4">
            <Button
              variant="outline"
              size="sm"
              onClick={() => onFiltersChange({
                showRegions: true,
                showV2GCalls: true,
                showGridInvitations: true,
                showVehicles: true,
                showScoreBubbles: true,
                showHotZones: true,
                showCharging: true,
                showDischarging: true,
                showIdle: true,
                showOffline: true,
                showSocInfo: true
              })}
              className="flex-1"
              data-testid="button-show-all"
            >
              Show All
            </Button>
            <Button
              variant="outline"
              size="sm"
              onClick={() => onFiltersChange({
                showRegions: false,
                showV2GCalls: false,
                showGridInvitations: false,
                showVehicles: true,
                showScoreBubbles: false,
                showHotZones: false,
                showCharging: true,
                showDischarging: true,
                showIdle: false,
                showOffline: false,
                showSocInfo: true
              })}
              className="flex-1"
              data-testid="button-essentials-only"
            >
              Essentials
            </Button>
          </div>
        </div>
      </SheetContent>
    </Sheet>
  );
}