import { useEffect, useMemo } from "react";
import { useMap } from "react-leaflet";
import L from "leaflet";
import { GridInvitation } from "@shared/schema";

interface GridInvitationLayerProps {
  invitations: GridInvitation[];
  visible: boolean;
  timeWindowMinutes?: number;
}

// Get styling based on demand level and priority
const getInvitationStyle = (demandLevel: string, priority: number) => {
  const baseStyle = {
    weight: 3,
    opacity: 0.9,
    fillOpacity: 0.7,
    className: 'grid-invitation'
  };

  switch (demandLevel.toLowerCase()) {
    case 'peak':
      return {
        ...baseStyle,
        color: '#dc2626', // red for peak demand
        fillColor: '#fca5a5',
        className: 'grid-invitation-peak'
      };
    case 'high':
      return {
        ...baseStyle,
        color: '#ea580c', // orange for high demand
        fillColor: '#fed7aa',
        className: 'grid-invitation-high'
      };
    case 'medium':
      return {
        ...baseStyle,
        color: '#d97706', // amber for medium demand
        fillColor: '#fde68a',
        className: 'grid-invitation-medium'
      };
    default:
      return {
        ...baseStyle,
        color: '#059669', // emerald for normal
        fillColor: '#a7f3d0',
        className: 'grid-invitation-normal'
      };
  }
};

// Get invitation icon based on demand level
const createInvitationIcon = (demandLevel: string, priority: number, incentive?: number) => {
  const style = getInvitationStyle(demandLevel, priority);
  
  return L.divIcon({
    html: `
      <div class="invitation-marker flex flex-col items-center">
        <div class="invitation-pulse w-8 h-8 rounded-full border-2 flex items-center justify-center text-white font-bold text-xs" 
             style="background-color: ${style.fillColor}; border-color: ${style.color}; animation: invitation-pulse 2s infinite;">
          ${priority}
        </div>
        <div class="invitation-label bg-white dark:bg-gray-800 px-2 py-1 rounded shadow text-xs font-medium border mt-1"
             style="border-color: ${style.color};">
          <div style="color: ${style.color};" class="flex items-center gap-1">
            <span>🔌</span>
            <span>${demandLevel.toUpperCase()}</span>
          </div>
          ${incentive ? `<div class="text-green-600">+$${incentive.toFixed(2)}</div>` : ''}
        </div>
      </div>
    `,
    className: 'grid-invitation-icon',
    iconSize: [60, 80],
    iconAnchor: [30, 40]
  });
};

export default function GridInvitationLayer({ 
  invitations, 
  visible, 
  timeWindowMinutes = 120 
}: GridInvitationLayerProps) {
  const map = useMap();

  const activeInvitations = useMemo(() => {
    const now = new Date();
    return invitations.filter(invitation => {
      const expiresAt = new Date(invitation.expiresAt);
      const createdAt = new Date(invitation.timestamp);
      const minutesAgo = (now.getTime() - createdAt.getTime()) / (1000 * 60);
      
      return minutesAgo <= timeWindowMinutes && expiresAt > now;
    });
  }, [invitations, timeWindowMinutes]);

  useEffect(() => {
    if (!visible || activeInvitations.length === 0) return;

    const invitationLayer = L.layerGroup();

    // Add CSS for invitation animations
    const style = document.createElement('style');
    style.textContent = `
      .invitation-pulse {
        animation: invitation-pulse 2s infinite;
      }
      
      .grid-invitation-peak .invitation-pulse {
        animation: invitation-pulse-urgent 1s infinite;
      }
      
      .grid-invitation-high .invitation-pulse {
        animation: invitation-pulse-fast 1.5s infinite;
      }
      
      @keyframes invitation-pulse {
        0% { transform: scale(1); box-shadow: 0 0 0 0 rgba(239, 68, 68, 0.7); }
        50% { transform: scale(1.1); box-shadow: 0 0 0 8px rgba(239, 68, 68, 0.3); }
        100% { transform: scale(1); box-shadow: 0 0 0 0 rgba(239, 68, 68, 0); }
      }
      
      @keyframes invitation-pulse-urgent {
        0% { transform: scale(1); box-shadow: 0 0 0 0 rgba(220, 38, 38, 0.8); }
        50% { transform: scale(1.2); box-shadow: 0 0 0 12px rgba(220, 38, 38, 0.4); }
        100% { transform: scale(1); box-shadow: 0 0 0 0 rgba(220, 38, 38, 0); }
      }
      
      @keyframes invitation-pulse-fast {
        0% { transform: scale(1); box-shadow: 0 0 0 0 rgba(234, 88, 12, 0.7); }
        50% { transform: scale(1.15); box-shadow: 0 0 0 10px rgba(234, 88, 12, 0.3); }
        100% { transform: scale(1); box-shadow: 0 0 0 0 rgba(234, 88, 12, 0); }
      }
    `;
    
    if (!document.querySelector('#grid-invitation-styles')) {
      style.id = 'grid-invitation-styles';
      document.head.appendChild(style);
    }

    activeInvitations.forEach(invitation => {
      const icon = createInvitationIcon(
        invitation.demandLevel, 
        invitation.priority, 
        invitation.incentiveAmount || undefined
      );
      
      const marker = L.marker([invitation.lat, invitation.lng], { icon });

      // Add popup with invitation details
      const now = new Date();
      const expiresAt = new Date(invitation.expiresAt);
      const minutesLeft = Math.round((expiresAt.getTime() - now.getTime()) / (1000 * 60));
      
      const popupContent = `
        <div class="text-sm space-y-2 min-w-[220px]" data-testid="popup-grid-invitation-${invitation.id}">
          <div class="flex items-center gap-2 font-semibold text-red-600">
            <div class="w-3 h-3 bg-red-500 rounded-full animate-pulse"></div>
            <span>Grid Connection Invitation</span>
          </div>
          <div class="space-y-1">
            <div class="flex justify-between">
              <span>Demand Level:</span>
              <span class="font-medium uppercase">${invitation.demandLevel}</span>
            </div>
            <div class="flex justify-between">
              <span>Priority:</span>
              <span class="font-medium">${invitation.priority}/10</span>
            </div>
            ${invitation.incentiveAmount ? `
              <div class="flex justify-between">
                <span>Incentive:</span>
                <span class="font-medium text-green-600">+$${invitation.incentiveAmount.toFixed(2)}</span>
              </div>
            ` : ''}
            <div class="flex justify-between">
              <span>Region:</span>
              <span class="font-medium">${invitation.region || 'N/A'}</span>
            </div>
            <div class="flex justify-between">
              <span>Expires:</span>
              <span class="font-medium ${minutesLeft <= 10 ? 'text-red-600' : ''}">${minutesLeft}m</span>
            </div>
            <div class="flex justify-between">
              <span>Location:</span>
              <span class="font-medium text-xs">${invitation.lat.toFixed(4)}, ${invitation.lng.toFixed(4)}</span>
            </div>
          </div>
          <div class="text-xs mt-2 p-2 rounded" style="background: linear-gradient(to right, #fee2e2, #fef3c7);">
            🚗 <strong>CONNECT NOW:</strong> The grid is requesting EV connections at this location to help balance ${invitation.demandLevel} demand.
            ${invitation.incentiveAmount ? ` Earn $${invitation.incentiveAmount.toFixed(2)} for participating!` : ''}
          </div>
        </div>
      `;

      marker.bindPopup(popupContent, {
        className: 'grid-invitation-popup'
      });

      marker.on('click', () => {
        console.log(`Grid invitation selected:`, invitation);
      });

      invitationLayer.addLayer(marker);
    });

    invitationLayer.addTo(map);

    return () => {
      map.removeLayer(invitationLayer);
    };
  }, [map, activeInvitations, visible]);

  return null;
}