import { useState } from "react";
import { Link, useLocation } from "wouter";
import { 
  Map, 
  Mail, 
  DollarSign, 
  Settings, 
  Bell,
  Menu
} from "lucide-react";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { cn } from "@/lib/utils";

interface MobileShellProps {
  children: React.ReactNode;
  unreadPings?: number;
  showNotificationBanner?: boolean;
}

export function MobileShell({ 
  children, 
  unreadPings = 0,
  showNotificationBanner = false 
}: MobileShellProps) {
  const [location] = useLocation();
  const [showMobileMenu, setShowMobileMenu] = useState(false);

  const tabs = [
    {
      id: "map",
      label: "Map",
      icon: Map,
      path: "/",
      active: location === "/" || location.startsWith("/overview")
    },
    {
      id: "invites",
      label: "Invites", 
      icon: Mail,
      path: "/grid-requirements",
      active: location.startsWith("/grid-requirements"),
      badge: unreadPings > 0 ? unreadPings : undefined
    },
    {
      id: "earnings",
      label: "Earnings",
      icon: DollarSign, 
      path: "/incentive-simulation",
      active: location.startsWith("/incentive-simulation")
    },
    {
      id: "settings",
      label: "Settings",
      icon: Settings,
      path: "/settings", 
      active: location.startsWith("/settings")
    }
  ];

  return (
    <div className="flex flex-col h-screen bg-background">
      {/* Top App Bar */}
      <header className="flex items-center justify-between p-4 border-b bg-card/50 backdrop-blur-sm">
        <div className="flex items-center gap-3">
          <Button
            variant="ghost"
            size="icon"
            onClick={() => setShowMobileMenu(!showMobileMenu)}
            data-testid="button-mobile-menu"
            className="h-8 w-8"
          >
            <Menu className="h-4 w-4" />
          </Button>
          <h1 className="font-semibold text-lg">V2G Platform</h1>
        </div>
        
        <div className="flex items-center gap-2">
          {/* Notification Bell */}
          <Button
            variant="ghost" 
            size="icon"
            className="h-8 w-8 relative"
            data-testid="button-notifications"
          >
            <Bell className="h-4 w-4" />
            {unreadPings > 0 && (
              <Badge 
                variant="destructive" 
                className="absolute -top-1 -right-1 h-5 w-5 text-xs p-0 flex items-center justify-center"
                data-testid={`badge-ping-count-${unreadPings}`}
              >
                {unreadPings > 9 ? "9+" : unreadPings}
              </Badge>
            )}
          </Button>
        </div>
      </header>

      {/* Main Content Area */}
      <main className="flex-1 overflow-hidden relative">
        {children}
      </main>

      {/* Bottom Tab Navigation */}
      <nav className="border-t bg-card/80 backdrop-blur-sm">
        <div className="flex items-center justify-around px-2 py-1">
          {tabs.map((tab) => {
            const Icon = tab.icon;
            return (
              <Link key={tab.id} href={tab.path}>
                <Button
                  variant="ghost"
                  className={cn(
                    "flex flex-col items-center gap-1 h-auto py-2 px-3 min-w-[4rem]",
                    tab.active 
                      ? "text-primary bg-primary/10" 
                      : "text-muted-foreground hover:text-foreground"
                  )}
                  data-testid={`tab-${tab.id}`}
                >
                  <div className="relative">
                    <Icon className="h-5 w-5" />
                    {tab.badge && (
                      <Badge 
                        variant="destructive" 
                        className="absolute -top-2 -right-2 h-4 w-4 text-xs p-0 flex items-center justify-center"
                        data-testid={`badge-${tab.id}-${tab.badge}`}
                      >
                        {tab.badge > 9 ? "9+" : tab.badge}
                      </Badge>
                    )}
                  </div>
                  <span className="text-xs font-medium">{tab.label}</span>
                </Button>
              </Link>
            );
          })}
        </div>
      </nav>
    </div>
  );
}