import { useEffect, useMemo } from "react";
import { useMap } from "react-leaflet";
import L from "leaflet";
import { Vehicle } from "./VehicleMap";
import { RegionStat } from "@shared/schema";

interface RegionLayerProps {
  vehicles: Vehicle[];
  onRegionSelect?: (region: RegionStat) => void;
  visible: boolean;
}

// Grid cell size based on zoom level
const getGridSize = (zoom: number): number => {
  if (zoom <= 8) return 0.1; // ~11km cells
  if (zoom <= 10) return 0.05; // ~5.5km cells  
  if (zoom <= 12) return 0.02; // ~2.2km cells
  return 0.01; // ~1.1km cells
};

// Generate grid cell ID from coordinates
const getGridCellId = (lat: number, lng: number, gridSize: number): string => {
  const gridLat = Math.floor(lat / gridSize) * gridSize;
  const gridLng = Math.floor(lng / gridSize) * gridSize;
  return `${gridLat.toFixed(4)},${gridLng.toFixed(4)}`;
};

// Calculate regional statistics from vehicles
const calculateRegionStats = (vehicles: Vehicle[], gridSize: number): RegionStat[] => {
  const regionMap = new Map<string, {
    vehicles: Vehicle[];
    bounds: { north: number; south: number; east: number; west: number };
  }>();

  // Group vehicles by grid cell
  vehicles.forEach(vehicle => {
    const cellId = getGridCellId(vehicle.lat, vehicle.lng, gridSize);
    
    if (!regionMap.has(cellId)) {
      const [latStr, lngStr] = cellId.split(',');
      const lat = parseFloat(latStr);
      const lng = parseFloat(lngStr);
      
      regionMap.set(cellId, {
        vehicles: [],
        bounds: {
          north: lat + gridSize,
          south: lat,
          east: lng + gridSize,
          west: lng
        }
      });
    }
    
    regionMap.get(cellId)!.vehicles.push(vehicle);
  });

  // Calculate statistics for each region
  return Array.from(regionMap.entries()).map(([regionId, data]) => {
    const { vehicles: regionVehicles, bounds } = data;
    
    const statusCounts = regionVehicles.reduce((acc, v) => {
      const status = v.status.toLowerCase();
      acc[status] = (acc[status] || 0) + 1;
      return acc;
    }, {} as Record<string, number>);

    const totalCapacity = regionVehicles.reduce((sum, v) => sum + v.battery, 0);
    const averageSOC = regionVehicles.length > 0 
      ? Math.round(regionVehicles.reduce((sum, v) => sum + v.soc, 0) / regionVehicles.length)
      : 0;

    return {
      regionId,
      bounds,
      vehicleCount: regionVehicles.length,
      chargingCount: statusCounts.charging || 0,
      dischargingCount: statusCounts.discharging || 0,
      idleCount: statusCounts.idle || 0,
      offlineCount: statusCounts.offline || 0,
      totalCapacity,
      averageSOC
    };
  });
};

// Get color based on charging ratio
const getRegionColor = (chargingCount: number, totalCount: number): string => {
  if (totalCount === 0) return '#6b7280'; // gray
  
  const ratio = chargingCount / totalCount;
  if (ratio >= 0.7) return '#22c55e'; // high charging - green
  if (ratio >= 0.4) return '#3b82f6'; // medium charging - blue
  if (ratio >= 0.1) return '#f59e0b'; // low charging - yellow
  return '#ef4444'; // no charging - red
};

export default function RegionLayer({ vehicles, onRegionSelect, visible }: RegionLayerProps) {
  const map = useMap();

  const { zoom, gridSize } = useMemo(() => {
    const currentZoom = map.getZoom();
    return {
      zoom: currentZoom,
      gridSize: getGridSize(currentZoom)
    };
  }, [map]);

  const regionStats = useMemo(() => {
    return calculateRegionStats(vehicles, gridSize);
  }, [vehicles, gridSize]);

  useEffect(() => {
    if (!visible) return;

    const regionLayer = L.layerGroup();

    regionStats.forEach(region => {
      const { bounds, chargingCount, vehicleCount, regionId } = region;
      
      // Create rectangle for region
      const rectangle = L.rectangle(
        [[bounds.south, bounds.west], [bounds.north, bounds.east]],
        {
          color: getRegionColor(chargingCount, vehicleCount),
          weight: 2,
          opacity: 0.8,
          fillOpacity: 0.2,
          className: 'region-cell'
        }
      );

      // Add click handler
      rectangle.on('click', () => {
        console.log(`Selected region: ${regionId}`);
        onRegionSelect?.(region);
      });

      // Add tooltip with region stats
      const tooltipContent = `
        <div class="text-sm space-y-1">
          <div class="font-semibold">Region ${regionId}</div>
          <div>Total Vehicles: ${vehicleCount}</div>
          <div class="text-green-600">Charging: ${chargingCount}</div>
          <div class="text-blue-600">Discharging: ${region.dischargingCount}</div>
          <div class="text-gray-600">Idle: ${region.idleCount}</div>
          <div class="text-red-600">Offline: ${region.offlineCount}</div>
          <div>Avg SOC: ${region.averageSOC}%</div>
        </div>
      `;

      rectangle.bindTooltip(tooltipContent, {
        sticky: true,
        className: 'region-tooltip'
      });

      // Add label for charging count if significant
      if (chargingCount > 0) {
        const center = rectangle.getBounds().getCenter();
        const label = L.marker(center, {
          icon: L.divIcon({
            html: `<div class="region-label bg-white dark:bg-gray-800 px-2 py-1 rounded shadow text-xs font-bold border">
              ⚡ ${chargingCount}
            </div>`,
            className: 'region-label-marker',
            iconSize: [40, 20],
            iconAnchor: [20, 10]
          })
        });
        
        regionLayer.addLayer(label);
      }

      regionLayer.addLayer(rectangle);
    });

    regionLayer.addTo(map);

    return () => {
      map.removeLayer(regionLayer);
    };
  }, [map, regionStats, visible, onRegionSelect]);

  return null;
}