import { useEffect, useRef } from "react";
import { useMap } from "react-leaflet";
import L from "leaflet";
import { EVScoring } from "@shared/schema";
import { Vehicle } from "./VehicleMap";

interface ScoreBubblesLayerProps {
  vehicleScores: EVScoring[];
  vehicles: Vehicle[]; // Real vehicle data for positions
  visible: boolean;
}

export default function ScoreBubblesLayer({ vehicleScores, vehicles, visible }: ScoreBubblesLayerProps) {
  const map = useMap();
  const bubblesRef = useRef<L.CircleMarker[]>([]);

  useEffect(() => {
    console.log('ScoreBubblesLayer useEffect triggered. visible:', visible, 'currentBubbleCount:', bubblesRef.current.length);
    
    // Always clear existing bubbles first
    const bubblesToRemove = bubblesRef.current.length;
    bubblesRef.current.forEach((bubble, index) => {
      if (map.hasLayer(bubble)) {
        map.removeLayer(bubble);
        console.log(`Removed bubble ${index + 1}/${bubblesToRemove} from map`);
      } else {
        console.log(`Bubble ${index + 1}/${bubblesToRemove} was not on map`);
      }
    });
    bubblesRef.current = [];
    console.log('Cleared all bubbles. Ref now contains:', bubblesRef.current.length, 'bubbles');

    // Only add new bubbles if visible and we have data
    if (!visible || !vehicles || vehicles.length === 0) {
      console.log('ScoreBubblesLayer: bubbles hidden or no data. visible:', visible, 'vehicleCount:', vehicles?.length || 0);
      return;
    }

    console.log('ScoreBubblesLayer: rendering bubbles. visible:', visible, 'vehicleCount:', vehicles.length);

    vehicleScores.forEach((score) => {
      // Find matching vehicle by name (scoring uses vehicle names like "EV-101")
      const vehicle = vehicles.find(v => v.name === score.vehicleId);
      if (!vehicle) return;

      // Calculate bubble size based on eligibility score (10-50px radius)
      const radius = Math.max(10, Math.min(50, (score.eligibilityScore / 100) * 40 + 10));
      
      // Color based on score (red: 0-30, yellow: 30-70, green: 70-100)
      const getScoreColor = (score: number) => {
        if (score >= 70) return '#22c55e'; // green
        if (score >= 30) return '#f59e0b'; // yellow
        return '#ef4444'; // red
      };

      const bubble = L.circleMarker([vehicle.lat, vehicle.lng], {
        radius: radius,
        fillColor: getScoreColor(score.eligibilityScore),
        color: getScoreColor(score.eligibilityScore),
        weight: 2,
        opacity: 0.6,
        fillOpacity: 0.3,
        className: 'eligibility-score-bubble'
      });

      // Add popup with scoring details
      bubble.bindPopup(`
        <div class="p-2 space-y-2">
          <div class="font-bold text-sm">${score.vehicleId}</div>
          <div class="text-xs">
            <div><strong>Score:</strong> ${score.eligibilityScore.toFixed(1)}/100</div>
            <div><strong>Surplus:</strong> ${score.surplusKWh.toFixed(1)} kWh</div>
            <div><strong>Max Power:</strong> ${score.maxDischargeKW.toFixed(1)} kW</div>
            <div class="mt-2 pt-2 border-t">
              <div><strong>Factors:</strong></div>
              <div>Availability: ${(score.factors.availability * 100).toFixed(0)}%</div>
              <div>SoC Buffer: ${(score.factors.socBuffer * 100).toFixed(0)}%</div>
              <div>Battery Health: ${score.factors.batteryHealth.toFixed(0)}%</div>
              <div>Location Priority: ${(score.factors.locationPriority * 100).toFixed(0)}%</div>
            </div>
          </div>
        </div>
      `);

      bubble.addTo(map);
      bubblesRef.current.push(bubble);
      console.log(`Added bubble for ${score.vehicleId} at [${vehicle.lat}, ${vehicle.lng}]. Total bubbles: ${bubblesRef.current.length}`);
    });

    // Cleanup function for when component unmounts
    return () => {
      bubblesRef.current.forEach(bubble => {
        if (map.hasLayer(bubble)) {
          map.removeLayer(bubble);
        }
      });
      bubblesRef.current = [];
    };
  }, [map, vehicleScores, vehicles, visible]);

  return null;
}

