import { useEffect, useMemo } from "react";
import { useMap } from "react-leaflet";
import L from "leaflet";
import { V2GCall } from "@shared/schema";

interface V2GBlipsLayerProps {
  v2gCalls: V2GCall[];
  visible: boolean;
  timeWindowMinutes?: number; // Only show calls from last N minutes
}

// Get time-based styling for V2G call blips
const getBlipStyle = (timestamp: string, timeWindowMinutes: number = 60) => {
  const now = new Date();
  const callTime = new Date(timestamp);
  const minutesAgo = (now.getTime() - callTime.getTime()) / (1000 * 60);
  
  if (minutesAgo <= 5) {
    // Very recent - bright pulsing
    return {
      color: '#3b82f6', // electric blue
      weight: 4,
      opacity: 1,
      fillOpacity: 0.8,
      className: 'v2g-blip-recent'
    };
  } else if (minutesAgo <= 15) {
    // Recent - medium pulse
    return {
      color: '#06b6d4', // cyan
      weight: 3,
      opacity: 0.8,
      fillOpacity: 0.6,
      className: 'v2g-blip-medium'
    };
  } else if (minutesAgo <= timeWindowMinutes) {
    // Older - subtle
    return {
      color: '#8b5cf6', // purple
      weight: 2,
      opacity: 0.6,
      fillOpacity: 0.4,
      className: 'v2g-blip-old'
    };
  }
  
  return null; // Too old, don't show
};

// Get blip size based on magnitude
const getBlipRadius = (magnitude: number): number => {
  return Math.max(8, Math.min(20, magnitude * 2));
};

export default function V2GBlipsLayer({ 
  v2gCalls, 
  visible, 
  timeWindowMinutes = 60 
}: V2GBlipsLayerProps) {
  const map = useMap();

  const recentCalls = useMemo(() => {
    const now = new Date();
    return v2gCalls.filter(call => {
      const callTime = new Date(call.timestamp);
      const minutesAgo = (now.getTime() - callTime.getTime()) / (1000 * 60);
      return minutesAgo <= timeWindowMinutes;
    });
  }, [v2gCalls, timeWindowMinutes]);

  useEffect(() => {
    if (!visible || recentCalls.length === 0) return;

    const blipsLayer = L.layerGroup();

    // Add CSS for pulsing animation
    const style = document.createElement('style');
    style.textContent = `
      .v2g-blip-recent {
        animation: pulse-strong 1.5s infinite;
      }
      .v2g-blip-medium {
        animation: pulse-medium 2s infinite;
      }
      .v2g-blip-old {
        animation: pulse-subtle 3s infinite;
      }
      
      @keyframes pulse-strong {
        0% { transform: scale(1); opacity: 1; }
        50% { transform: scale(1.4); opacity: 0.7; }
        100% { transform: scale(1); opacity: 1; }
      }
      
      @keyframes pulse-medium {
        0% { transform: scale(1); opacity: 0.8; }
        50% { transform: scale(1.3); opacity: 0.5; }
        100% { transform: scale(1); opacity: 0.8; }
      }
      
      @keyframes pulse-subtle {
        0% { transform: scale(1); opacity: 0.6; }
        50% { transform: scale(1.2); opacity: 0.3; }
        100% { transform: scale(1); opacity: 0.6; }
      }
    `;
    
    if (!document.querySelector('#v2g-blips-styles')) {
      style.id = 'v2g-blips-styles';
      document.head.appendChild(style);
    }

    recentCalls.forEach(call => {
      const blipStyle = getBlipStyle(call.timestamp, timeWindowMinutes);
      if (!blipStyle) return;

      const radius = getBlipRadius(call.magnitude);
      
      // Create the main blip circle
      const blip = L.circleMarker([call.lat, call.lng], {
        ...blipStyle,
        radius
      });

      // Add popup with call details
      const now = new Date();
      const callTime = new Date(call.timestamp);
      const minutesAgo = Math.round((now.getTime() - callTime.getTime()) / (1000 * 60));
      
      const popupContent = `
        <div class="text-sm space-y-2 min-w-[200px]" data-testid="popup-v2g-call-${call.id}">
          <div class="flex items-center gap-2 font-semibold text-blue-600">
            <div class="w-3 h-3 bg-blue-500 rounded-full animate-pulse"></div>
            <span>V2G Aggregation Call</span>
          </div>
          <div class="space-y-1">
            <div class="flex justify-between">
              <span>Time:</span>
              <span class="font-medium">${minutesAgo}m ago</span>
            </div>
            <div class="flex justify-between">
              <span>Magnitude:</span>
              <span class="font-medium">${call.magnitude}/10</span>
            </div>
            <div class="flex justify-between">
              <span>Region:</span>
              <span class="font-medium">${call.region || 'N/A'}</span>
            </div>
            <div class="flex justify-between">
              <span>Location:</span>
              <span class="font-medium text-xs">${call.lat.toFixed(4)}, ${call.lng.toFixed(4)}</span>
            </div>
          </div>
          <div class="text-xs text-gray-600 mt-2 p-2 bg-blue-50 dark:bg-blue-900/20 rounded">
            Grid demand spike detected at this location. V2G resources were called to respond.
          </div>
        </div>
      `;

      blip.bindPopup(popupContent, {
        className: 'v2g-call-popup'
      });

      blip.on('click', () => {
        console.log(`V2G call selected:`, call);
      });

      blipsLayer.addLayer(blip);
    });

    blipsLayer.addTo(map);

    return () => {
      map.removeLayer(blipsLayer);
    };
  }, [map, recentCalls, visible, timeWindowMinutes]);

  return null;
}