import { useState } from "react";
import { Card, CardContent } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Tabs, TabsList, TabsTrigger, TabsContent } from "@/components/ui/tabs";
import { Badge } from "@/components/ui/badge";
import { Progress } from "@/components/ui/progress";
import { motion } from "framer-motion";
import { 
  Battery, 
  DollarSign, 
  Shield, 
  Send, 
  Car, 
  Zap,
  MapPin,
  AlertTriangle 
} from "lucide-react";
import { Vehicle } from "./VehicleMap";

interface VehicleInfoPanelProps {
  selectedVehicle: Vehicle | null;
  onDispatchSignal?: (vehicleId: number) => void;
  className?: string;
  showSocInfo?: boolean;
}

export default function VehicleInfoPanel({ 
  selectedVehicle, 
  onDispatchSignal,
  className = "",
  showSocInfo = true
}: VehicleInfoPanelProps) {
  const [activeTab, setActiveTab] = useState("overview");

  const getStatusColor = (status: string) => {
    switch (status.toLowerCase()) {
      case "charging":
        return "bg-vehicle-charging text-white";
      case "discharging": 
        return "bg-vehicle-discharging text-white";
      case "offline":
        return "bg-vehicle-error text-white";
      default:
        return "bg-vehicle-idle text-white";
    }
  };

  const getStatusIcon = (status: string) => {
    switch (status.toLowerCase()) {
      case "charging":
        return <Battery className="w-4 h-4" />;
      case "discharging":
        return <Zap className="w-4 h-4" />;
      default:
        return <Car className="w-4 h-4" />;
    }
  };

  const handleDispatch = () => {
    if (selectedVehicle && onDispatchSignal) {
      console.log(`Dispatching signal to ${selectedVehicle.name}`);
      onDispatchSignal(selectedVehicle.id);
    }
  };

  return (
    <Card className={`h-full flex flex-col ${className}`}>
      <CardContent className="flex-1 flex flex-col p-6">
        <Tabs value={activeTab} onValueChange={setActiveTab} className="flex-1 flex flex-col">
          <TabsList className="grid grid-cols-3 mb-6">
            <TabsTrigger value="overview" data-testid="tab-overview">Overview</TabsTrigger>
            <TabsTrigger value="criteria" data-testid="tab-criteria">Profile</TabsTrigger>
            <TabsTrigger value="benefits" data-testid="tab-benefits">Benefits</TabsTrigger>
          </TabsList>

          {/* Overview Tab */}
          <TabsContent value="overview" className="flex-1" data-testid="content-overview">
            {selectedVehicle ? (
              <motion.div 
                className="space-y-6" 
                initial={{ opacity: 0, y: 20 }} 
                animate={{ opacity: 1, y: 0 }}
                transition={{ duration: 0.3 }}
              >
                <div className="space-y-4">
                  <div className="flex items-center gap-3">
                    <Battery className="w-6 h-6 text-primary" />
                    <h2 className="text-xl font-bold" data-testid="text-vehicle-name">
                      {selectedVehicle.name}
                    </h2>
                    <Badge className={getStatusColor(selectedVehicle.status)}>
                      {getStatusIcon(selectedVehicle.status)}
                      <span className="ml-2">{selectedVehicle.status}</span>
                    </Badge>
                  </div>

                  <div className="space-y-4">
                    {showSocInfo && (
                      <div>
                        <div className="flex justify-between items-center mb-2">
                          <span className="text-sm font-medium">State of Charge</span>
                          <span className="text-sm font-bold" data-testid="text-soc">
                            {selectedVehicle.soc}%
                          </span>
                        </div>
                        <Progress 
                          value={selectedVehicle.soc} 
                          className="h-3"
                          data-testid="progress-soc"
                        />
                      </div>
                    )}

                    <div className="grid grid-cols-2 gap-4">
                      {showSocInfo && (
                        <div className="space-y-1">
                          <span className="text-sm text-muted-foreground">Battery Capacity</span>
                          <p className="font-semibold" data-testid="text-battery">
                            {selectedVehicle.battery} kWh
                          </p>
                        </div>
                      )}
                      <div className="space-y-1">
                        <span className="text-sm text-muted-foreground">Vehicle Type</span>
                        <p className="font-semibold" data-testid="text-fleet-status">
                          {selectedVehicle.fleet ? "Fleet Vehicle" : "Private EV"}
                        </p>
                      </div>
                    </div>
                  </div>

                  <div className="flex gap-2">
                    <Button 
                      onClick={handleDispatch}
                      className="flex-1" 
                      size="sm"
                      data-testid="button-dispatch"
                    >
                      <Send className="w-4 h-4 mr-2" />
                      Send Dispatch Signal
                    </Button>
                  </div>
                </div>
              </motion.div>
            ) : (
              <div className="flex flex-col items-center justify-center h-full text-center space-y-4">
                <MapPin className="w-12 h-12 text-muted-foreground" />
                <p className="text-muted-foreground" data-testid="text-no-selection">
                  Select a vehicle on the map to view details
                </p>
              </div>
            )}
          </TabsContent>

          {/* User Profile Tab */}
          <TabsContent value="criteria" className="flex-1 overflow-y-auto" data-testid="content-criteria">
            <div className="space-y-6">
              <div className="text-center space-y-4">
                <div className="w-16 h-16 bg-gradient-to-br from-green-500 to-green-600 rounded-full mx-auto flex items-center justify-center">
                  <span className="text-white font-bold text-lg">AJ</span>
                </div>
                <div>
                  <h3 className="font-bold text-lg">Alex Johnson</h3>
                  <Badge className="bg-green-500 text-white mt-2">
                    <Shield className="w-3 h-3 mr-1" />
                    Green Champion
                  </Badge>
                </div>
              </div>

              <div className="grid grid-cols-3 gap-4 text-center">
                <div className="space-y-1">
                  <div className="text-2xl font-bold text-primary">5</div>
                  <div className="text-xs text-muted-foreground">V2G Sessions</div>
                </div>
                <div className="space-y-1">
                  <div className="text-2xl font-bold text-green-600">250</div>
                  <div className="text-xs text-muted-foreground">kWh Supplied</div>
                </div>
                <div className="space-y-1">
                  <div className="text-2xl font-bold text-blue-600">92</div>
                  <div className="text-xs text-muted-foreground">Grid Score</div>
                </div>
              </div>


              <div className="space-y-4">
                <h4 className="font-semibold">Recent Achievements</h4>
                
                <div className="space-y-2">
                  <div className="flex items-center gap-3 p-2 rounded-lg bg-muted/50">
                    <div className="w-8 h-8 bg-yellow-500 rounded-full flex items-center justify-center">
                      <Zap className="w-4 h-4 text-white" />
                    </div>
                    <div className="flex-1">
                      <div className="text-sm font-medium">Peak Hero</div>
                      <div className="text-xs text-muted-foreground">Helped during 5 peak demand events</div>
                    </div>
                  </div>

                  <div className="flex items-center gap-3 p-2 rounded-lg bg-muted/50">
                    <div className="w-8 h-8 bg-green-500 rounded-full flex items-center justify-center">
                      <Battery className="w-4 h-4 text-white" />
                    </div>
                    <div className="flex-1">
                      <div className="text-sm font-medium">Eco Warrior</div>
                      <div className="text-xs text-muted-foreground">100+ V2G sessions completed</div>
                    </div>
                  </div>

                  <div className="flex items-center gap-3 p-2 rounded-lg bg-muted/50">
                    <div className="w-8 h-8 bg-blue-500 rounded-full flex items-center justify-center">
                      <Car className="w-4 h-4 text-white" />
                    </div>
                    <div className="flex-1">
                      <div className="text-sm font-medium">Grid Supporter</div>
                      <div className="text-xs text-muted-foreground">Contributed 1000+ kWh to grid</div>
                    </div>
                  </div>
                </div>
              </div>

              <div className="text-center text-xs text-muted-foreground mt-6">
                Member since January 2024 • Level 8 Contributor
              </div>
            </div>
          </TabsContent>

          {/* Benefits Tab */}
          <TabsContent value="benefits" className="flex-1" data-testid="content-benefits">
            <div className="space-y-4">
              <h4 className="font-semibold flex items-center gap-2">
                <DollarSign className="w-4 h-4" />
                Your Earnings & Benefits
              </h4>
              
              <div className="space-y-3">
                <div className="flex justify-between items-center p-3 bg-green-50 dark:bg-green-950/20 rounded-lg">
                  <div className="flex items-center gap-2">
                    <DollarSign className="w-4 h-4 text-green-600" />
                    <span className="text-sm font-medium">Wallet Balance</span>
                  </div>
                  <span className="font-bold text-green-600">$1,247.50</span>
                </div>

                <div className="flex justify-between items-center p-3 bg-blue-50 dark:bg-blue-950/20 rounded-lg">
                  <div className="flex items-center gap-2">
                    <Battery className="w-4 h-4 text-blue-600" />
                    <span className="text-sm font-medium">CO₂ Savings</span>
                  </div>
                  <span className="font-bold text-blue-600">2.8 tons</span>
                </div>

                <div className="flex justify-between items-center p-3 bg-purple-50 dark:bg-purple-950/20 rounded-lg">
                  <div className="flex items-center gap-2">
                    <Shield className="w-4 h-4 text-purple-600" />
                    <span className="text-sm font-medium">Gov Rebates</span>
                  </div>
                  <span className="font-bold text-purple-600">$380.00</span>
                </div>
              </div>
            </div>
          </TabsContent>
        </Tabs>
      </CardContent>
    </Card>
  );
}