import { useState, useEffect, useRef } from "react";
import { MapContainer, TileLayer, Marker, Popup } from "react-leaflet";
import { Card, CardContent } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import L from "leaflet";
import { Battery, Car, Zap, Layers, Eye, EyeOff } from "lucide-react";
import RegionLayer from "./RegionLayer";
import V2GBlipsLayer from "./V2GBlipsLayer";
import GridInvitationLayer from "./GridInvitationLayer";
import ScoreBubblesLayer from "./ScoreBubblesLayer";
import HotZonesLayer from "./HotZonesLayer";
import { RegionStat, V2GCall, GridInvitation, EVScoring, HotZone } from "@shared/schema";

// Custom marker icons for different vehicle states
const createVehicleIcon = (status: string) => {
  const colors = {
    charging: "#22c55e", // green
    discharging: "#3b82f6", // blue  
    idle: "#6b7280", // gray
    offline: "#ef4444" // red
  };
  
  return L.divIcon({
    html: `<div class="flex items-center justify-center w-8 h-8 rounded-full border-2 border-white shadow-lg" style="background-color: ${colors[status as keyof typeof colors] || colors.idle}">
      <svg width="16" height="16" viewBox="0 0 24 24" fill="white">
        <path d="M15.85 4.5l-.42.83a2 2 0 0 1-1.79 1.11h-3.28a2 2 0 0 1-1.79-1.11L8.15 4.5H2v15h20v-15h-6.15zM9 14.5l3-3 3 3M12 8.5v6"/>
      </svg>
    </div>`,
    className: 'vehicle-marker',
    iconSize: [32, 32],
    iconAnchor: [16, 16]
  });
};

export interface Vehicle {
  id: number;
  name: string;
  lat: number;
  lng: number;
  soc: number; // State of Charge percentage
  status: "Charging" | "Discharging" | "Idle" | "Offline";
  battery: number; // Battery capacity in kWh
  fleet: boolean;
}

interface VehicleMapProps {
  vehicles: Vehicle[];
  selectedVehicle: Vehicle | null;
  onVehicleSelect: (vehicle: Vehicle) => void;
  v2gCalls?: V2GCall[];
  gridInvitations?: GridInvitation[];
  vehicleScores?: EVScoring[];
  hotZones?: HotZone[];
  onRegionSelect?: (region: RegionStat) => void;
  onHotZoneSelect?: (zone: HotZone) => void;
  className?: string;
  showScoreBubbles?: boolean;
  showHotZones?: boolean;
  allowHotZoneEditing?: boolean;
  showSocInfo?: boolean;
}

export default function VehicleMap({ 
  vehicles, 
  selectedVehicle, 
  onVehicleSelect,
  v2gCalls = [],
  gridInvitations = [],
  vehicleScores = [],
  hotZones = [],
  onRegionSelect,
  onHotZoneSelect,
  className = "",
  showScoreBubbles = true,
  showHotZones = true,
  allowHotZoneEditing = false,
  showSocInfo = true
}: VehicleMapProps) {
  const [zoom, setZoom] = useState(12);
  const [showRegions, setShowRegions] = useState(true);
  const [showV2GCalls, setShowV2GCalls] = useState(true);
  const [showGridInvitations, setShowGridInvitations] = useState(true);
  const [showVehicles, setShowVehicles] = useState(true);
  const [showScoreBubblesState, setShowScoreBubblesState] = useState(showScoreBubbles);
  const [showHotZonesState, setShowHotZonesState] = useState(showHotZones);
  
  // Draggable position state - start below navigation area
  const [controlPosition, setControlPosition] = useState({ x: 16, y: 140 });
  const isDraggingRef = useRef(false);
  const dragOffsetRef = useRef({ x: 0, y: 0 });
  
  const getStatusIcon = (status: string) => {
    switch (status.toLowerCase()) {
      case "charging":
        return <Battery className="w-4 h-4 text-vehicle-charging" />;
      case "discharging":
        return <Zap className="w-4 h-4 text-vehicle-discharging" />;
      default:
        return <Car className="w-4 h-4 text-vehicle-idle" />;
    }
  };

  const getStatusColor = (status: string) => {
    switch (status.toLowerCase()) {
      case "charging":
        return "text-vehicle-charging";
      case "discharging": 
        return "text-vehicle-discharging";
      case "offline":
        return "text-vehicle-error";
      default:
        return "text-vehicle-idle";
    }
  };

  // Drag handlers using pointer events for better test compatibility
  const handlePointerDown = (e: React.PointerEvent) => {
    // Don't start dragging if clicking on a button
    if ((e.target as HTMLElement).closest('button')) {
      return;
    }
    
    const rect = e.currentTarget.getBoundingClientRect();
    dragOffsetRef.current = {
      x: e.clientX - rect.left,
      y: e.clientY - rect.top
    };
    isDraggingRef.current = true;
    e.currentTarget.setPointerCapture(e.pointerId);
  };

  const handlePointerMove = (e: React.PointerEvent) => {
    if (!isDraggingRef.current) return;

    const newX = e.clientX - dragOffsetRef.current.x;
    const newY = e.clientY - dragOffsetRef.current.y;
    
    // Constrain position to prevent overlapping with navigation
    const constrainedY = Math.max(120, Math.min(newY, window.innerHeight - 200));
    const constrainedX = Math.max(10, Math.min(newX, window.innerWidth - 250));
    
    setControlPosition({
      x: constrainedX,
      y: constrainedY
    });
  };

  const handlePointerUp = (e: React.PointerEvent) => {
    isDraggingRef.current = false;
    e.currentTarget.releasePointerCapture(e.pointerId);
  };

  // Determine which layers to show based on zoom level  
  const shouldShowRegions = showRegions;
  const shouldShowVehicles = showVehicles;

  return (
    <Card className={`h-full ${className}`}>
      {/* Draggable Layer Controls */}
      <div 
        className="absolute z-[1000] space-y-2 cursor-grab active:cursor-grabbing"
        style={{ 
          left: controlPosition.x, 
          top: controlPosition.y,
          userSelect: 'none',
          pointerEvents: 'auto',
          touchAction: 'none'
        }}
        onPointerDown={handlePointerDown}
        onPointerMove={handlePointerMove}
        onPointerUp={handlePointerUp}
        data-testid="draggable-controls"
      >
        <div className="bg-card/90 backdrop-blur-sm border rounded-lg p-2 shadow-lg">
          <div className="flex flex-col gap-2">
            <Button
              variant={showRegions ? "default" : "outline"}
              size="sm"
              onClick={() => setShowRegions(!showRegions)}
              data-testid="toggle-regions"
            >
              <Layers className="w-4 h-4 mr-2" />
              Regions
              {showRegions ? <Eye className="w-3 h-3 ml-2" /> : <EyeOff className="w-3 h-3 ml-2" />}
            </Button>
            
            <Button
              variant={showGridInvitations ? "default" : "outline"}
              size="sm"
              onClick={() => setShowGridInvitations(!showGridInvitations)}
              data-testid="toggle-grid-invitations"
            >
              <div className="w-4 h-4 mr-2 text-red-500">🔌</div>
              Grid Invites
              {showGridInvitations ? <Eye className="w-3 h-3 ml-2" /> : <EyeOff className="w-3 h-3 ml-2" />}
            </Button>
            
            <Button
              variant={showVehicles ? "default" : "outline"}
              size="sm"
              onClick={() => setShowVehicles(!showVehicles)}
              data-testid="toggle-vehicles"
            >
              <Car className="w-4 h-4 mr-2" />
              Vehicles
              {showVehicles ? <Eye className="w-3 h-3 ml-2" /> : <EyeOff className="w-3 h-3 ml-2" />}
            </Button>
            
            <Button
              variant={showScoreBubblesState ? "default" : "outline"}
              size="sm"
              onClick={() => {
                setShowScoreBubblesState(!showScoreBubblesState);
                console.log('Score bubbles toggled:', !showScoreBubblesState);
              }}
              data-testid="toggle-score-bubbles"
            >
              <div className="w-4 h-4 mr-2 text-blue-500">🎯</div>
              Scores
              {showScoreBubblesState ? <Eye className="w-3 h-3 ml-2" /> : <EyeOff className="w-3 h-3 ml-2" />}
            </Button>
            
            <Button
              variant={showHotZonesState ? "default" : "outline"}
              size="sm"
              onClick={() => setShowHotZonesState(!showHotZonesState)}
              data-testid="toggle-hot-zones"
            >
              <div className="w-4 h-4 mr-2 text-red-500">🔥</div>
              Hot Zones
              {showHotZonesState ? <Eye className="w-3 h-3 ml-2" /> : <EyeOff className="w-3 h-3 ml-2" />}
            </Button>
          </div>
        </div>
        
      </div>

      <CardContent className="h-full w-full p-0">
        <MapContainer 
          center={[1.3521, 103.8198]} 
          zoom={12} 
          className="h-full w-full rounded-lg"
          style={{ minHeight: "400px" }}
        >
          <TileLayer
            url="https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png"
          />
          
          {/* Regional layer for lower zoom levels */}
          {shouldShowRegions && (
            <RegionLayer 
              vehicles={vehicles} 
              onRegionSelect={onRegionSelect}
              visible={shouldShowRegions}
            />
          )}
          
          {/* V2G aggregation call blips */}
          {showV2GCalls && (
            <V2GBlipsLayer 
              v2gCalls={v2gCalls} 
              visible={showV2GCalls}
              timeWindowMinutes={60}
            />
          )}
          
          {/* Grid connection invitation blips */}
          {showGridInvitations && (
            <GridInvitationLayer 
              invitations={gridInvitations} 
              visible={showGridInvitations}
              timeWindowMinutes={120}
            />
          )}
          
          {/* Score bubbles around vehicles */}
          <ScoreBubblesLayer 
            vehicleScores={vehicleScores}
            vehicles={vehicles}
            visible={showScoreBubblesState}
          />
          
          {/* Hot zones for grid stress areas */}
          {showHotZonesState && (
            <HotZonesLayer 
              hotZones={hotZones}
              visible={showHotZonesState}
              onZoneSelect={onHotZoneSelect}
              editable={allowHotZoneEditing}
            />
          )}
          
          {/* Individual vehicle markers for higher zoom levels */}
          {shouldShowVehicles && vehicles.map((vehicle) => (
            <Marker 
              key={vehicle.id} 
              position={[vehicle.lat, vehicle.lng]}
              zIndexOffset={1000}
              eventHandlers={{ 
                click: (e) => {
                  // Ensure popup opens by manually handling it
                  e.target.openPopup();
                  onVehicleSelect(vehicle);
                }
              }}
            >
              <Popup 
                closeOnEscapeKey={true}
                closeOnClick={false}
              >
                <div className="text-sm space-y-2 min-w-[200px]" data-testid={`popup-vehicle-${vehicle.id}`}>
                  <div className="flex items-center gap-2 font-semibold">
                    {getStatusIcon(vehicle.status)}
                    <span>{vehicle.name}</span>
                  </div>
                  <div className="space-y-1">
                    <div className="flex justify-between">
                      <span>Status:</span>
                      <span className={`font-medium ${getStatusColor(vehicle.status)}`}>
                        {vehicle.status}
                      </span>
                    </div>
                    {showSocInfo && (
                      <>
                        <div className="flex justify-between">
                          <span>State of Charge:</span>
                          <span className="font-medium">{vehicle.soc}%</span>
                        </div>
                        <div className="flex justify-between">
                          <span>Battery:</span>
                          <span className="font-medium">{vehicle.battery} kWh</span>
                        </div>
                      </>
                    )}
                    <div className="flex justify-between">
                      <span>Type:</span>
                      <span className="font-medium">
                        {vehicle.fleet ? "Fleet Vehicle" : "Private EV"}
                      </span>
                    </div>
                  </div>
                </div>
              </Popup>
            </Marker>
          ))}
        </MapContainer>
      </CardContent>
    </Card>
  );
}