import { useState, useEffect, useCallback, createContext, useContext } from 'react';
import { useQuery, useMutation, useQueryClient } from '@tanstack/react-query';
import { apiRequest } from '@/lib/queryClient';
import { toast } from '@/hooks/use-toast';
import type { SelectPing, InsertNotificationSubscription } from '@shared/schema';

interface NotificationContextType {
  pings: SelectPing[];
  unreadCount: number;
  isSupported: boolean;
  isSubscribed: boolean;
  permission: NotificationPermission;
  subscribe: () => Promise<void>;
  unsubscribe: () => Promise<void>;
  dismissPing: (pingId: string) => void;
  markAsRead: (pingId: string) => void;
  requestPermission: () => Promise<NotificationPermission>;
}

const NotificationContext = createContext<NotificationContextType | null>(null);

interface NotificationProviderProps {
  children: React.ReactNode;
  userId?: string;
}

export function NotificationProvider({ children, userId = 'demo-user' }: NotificationProviderProps) {
  const [isSupported] = useState(() => 'Notification' in window && 'serviceWorker' in navigator);
  const [permission, setPermission] = useState<NotificationPermission>(() => 
    isSupported ? Notification.permission : 'denied'
  );
  const [isSubscribed, setIsSubscribed] = useState(false);
  const [dismissedPings, setDismissedPings] = useState<Set<string>>(new Set());
  const queryClient = useQueryClient();

  // Fetch recent pings
  const { data: pingsResponse = { pings: [] }, isError } = useQuery({
    queryKey: ['/api/pings/user', userId],
    enabled: !!userId,
    refetchInterval: 30000, // Poll every 30 seconds for new pings
  });

  // Extract pings array from API response shape { pings: SelectPing[] }
  const pingsData = (pingsResponse as any)?.pings || [];
  const pings = pingsData.filter((ping: SelectPing) => !dismissedPings.has(ping.id));
  const unreadCount = pings.length;

  // Subscribe to notifications
  const subscribeMutation = useMutation({
    mutationFn: async () => {
      if (!isSupported) throw new Error('Notifications not supported');
      
      const registration = await navigator.serviceWorker.ready;
      const subscription = await registration.pushManager.subscribe({
        userVisibleOnly: true,
        applicationServerKey: urlBase64ToUint8Array(
          // VAPID public key would go here in production
          'BMxPFQ5fvPW7JF9LLjrCdW-mqTCFWL_wnxKX8-JtALMjI0gJ_X9TQFfS4mZzXhD0C7rRyoqrXIe7CcvS_Cm3Wxg'
        )
      });

      const subscriptionData: InsertNotificationSubscription = {
        userId,
        endpoint: subscription.endpoint,
        p256dh: arrayBufferToBase64(subscription.getKey('p256dh')!),
        auth: arrayBufferToBase64(subscription.getKey('auth')!),
        platform: 'web'
      };

      await apiRequest('/api/notifications/subscribe', {
        method: 'POST',
        body: subscriptionData
      });

      return subscription;
    },
    onSuccess: () => {
      setIsSubscribed(true);
      toast({
        title: "Notifications Enabled",
        description: "You'll receive V2G alerts and updates"
      });
    },
    onError: (error) => {
      console.error('Failed to subscribe to notifications:', error);
      toast({
        title: "Notification Setup Failed",
        description: "Unable to enable push notifications",
        variant: "destructive"
      });
    }
  });

  // Unsubscribe from notifications
  const unsubscribeMutation = useMutation({
    mutationFn: async () => {
      const registration = await navigator.serviceWorker.ready;
      const subscription = await registration.pushManager.getSubscription();
      
      if (subscription) {
        await subscription.unsubscribe();
      }

      await apiRequest(`/api/notifications/subscribe/${userId}`, {
        method: 'DELETE'
      });
    },
    onSuccess: () => {
      setIsSubscribed(false);
      toast({
        title: "Notifications Disabled",
        description: "You won't receive push notifications"
      });
    }
  });

  // Register service worker and check subscription status
  useEffect(() => {
    if (!isSupported) return;

    const registerServiceWorker = async () => {
      try {
        const registration = await navigator.serviceWorker.register('/service-worker.js');
        console.log('Service Worker registered:', registration);

        // Check if already subscribed
        const subscription = await registration.pushManager.getSubscription();
        setIsSubscribed(!!subscription);

        // Handle foreground messages and WebSocket updates
        navigator.serviceWorker.addEventListener('message', (event) => {
          const { type, data } = event.data;
          if (type === 'PUSH_RECEIVED' || type === 'PING_CREATED') {
            // Handle in-app notification
            queryClient.invalidateQueries({ queryKey: ['/api/pings/user', userId] });
            
            toast({
              title: data.title || "New V2G Alert",
              description: data.message || data.body,
            });
          }
        });

      } catch (error) {
        console.error('Service Worker registration failed:', error);
      }
    };

    registerServiceWorker();
  }, [isSupported, userId, queryClient]);

  const requestPermission = useCallback(async (): Promise<NotificationPermission> => {
    if (!isSupported) return 'denied';
    
    const result = await Notification.requestPermission();
    setPermission(result);
    return result;
  }, [isSupported]);

  const subscribe = useCallback(async () => {
    if (permission !== 'granted') {
      const newPermission = await requestPermission();
      if (newPermission !== 'granted') {
        throw new Error('Notification permission denied');
      }
    }
    await subscribeMutation.mutateAsync();
  }, [permission, requestPermission, subscribeMutation]);

  const unsubscribe = useCallback(async () => {
    await unsubscribeMutation.mutateAsync();
  }, [unsubscribeMutation]);

  const dismissPing = useCallback((pingId: string) => {
    setDismissedPings(prev => new Set(Array.from(prev).concat([pingId])));
  }, []);

  const markAsRead = useCallback((pingId: string) => {
    // In a real app, this would update the server
    dismissPing(pingId);
  }, [dismissPing]);

  const value: NotificationContextType = {
    pings,
    unreadCount,
    isSupported,
    isSubscribed,
    permission,
    subscribe,
    unsubscribe,
    dismissPing,
    markAsRead,
    requestPermission,
  };

  return (
    <NotificationContext.Provider value={value}>
      {children}
    </NotificationContext.Provider>
  );
}

export function useNotifications() {
  const context = useContext(NotificationContext);
  if (!context) {
    throw new Error('useNotifications must be used within a NotificationProvider');
  }
  return context;
}

// Helper functions
function urlBase64ToUint8Array(base64String: string): Uint8Array {
  const padding = '='.repeat((4 - base64String.length % 4) % 4);
  const base64 = (base64String + padding).replace(/-/g, '+').replace(/_/g, '/');
  const rawData = window.atob(base64);
  const outputArray = new Uint8Array(rawData.length);
  for (let i = 0; i < rawData.length; ++i) {
    outputArray[i] = rawData.charCodeAt(i);
  }
  return outputArray;
}

function arrayBufferToBase64(buffer: ArrayBuffer): string {
  const bytes = new Uint8Array(buffer);
  let binary = '';
  for (let i = 0; i < bytes.byteLength; i++) {
    binary += String.fromCharCode(bytes[i]);
  }
  return window.btoa(binary);
}