import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Progress } from "@/components/ui/progress";
import { 
  Battery, 
  Zap, 
  TrendingUp, 
  TrendingDown,
  Clock,
  BarChart3,
  Activity,
  Gauge
} from "lucide-react";
import { 
  LineChart, 
  Line, 
  XAxis, 
  YAxis, 
  CartesianGrid, 
  Tooltip, 
  ResponsiveContainer,
  BarChart,
  Bar,
  PieChart,
  Pie,
  Cell
} from "recharts";

interface BatteryAnalytics {
  batteryType: string;
  count: number;
  avgCapacity: number; // kWh
  avgSOC: number; // %
  fastChargingCapable: boolean;
  maxDischargeRate: number; // kW
  cycleCount: number;
  healthScore: number; // %
}

interface ChargingSession {
  timestamp: string;
  chargingPower: number; // kW
  dischargingPower: number; // kW
  netFlow: number; // kW (positive = grid to vehicles, negative = vehicles to grid)
  peakDemandOffset: number; // kW offset provided to grid
  totalVehicles: number;
}

interface V2GSessionLimits {
  userLimit: number; // Max 5 per user
  manufacturerLimit: number; // Max 12 per car maker
  kwhPerSession: number; // 50 KWh per session
}

// Mock battery analytics data
const mockBatteryAnalytics: BatteryAnalytics[] = [
  {
    batteryType: "LFP (Lithium Iron Phosphate)",
    count: 45,
    avgCapacity: 75,
    avgSOC: 82,
    fastChargingCapable: true,
    maxDischargeRate: 8.5,
    cycleCount: 1250,
    healthScore: 94
  },
  {
    batteryType: "NCM (Nickel Cobalt Manganese)",
    count: 32,
    avgCapacity: 88,
    avgSOC: 78,
    fastChargingCapable: true,
    maxDischargeRate: 11.2,
    cycleCount: 980,
    healthScore: 96
  },
  {
    batteryType: "NCA (Nickel Cobalt Aluminum)",
    count: 18,
    avgCapacity: 95,
    avgSOC: 85,
    fastChargingCapable: true,
    maxDischargeRate: 12.0,
    cycleCount: 750,
    healthScore: 98
  },
  {
    batteryType: "LTO (Lithium Titanate)",
    count: 8,
    avgCapacity: 45,
    avgSOC: 90,
    fastChargingCapable: false,
    maxDischargeRate: 6.0,
    cycleCount: 3500,
    healthScore: 99
  }
];

// Mock charging session data for the last 24 hours
const mockChargingSessions: ChargingSession[] = Array.from({length: 24}, (_, i) => {
  const hour = i;
  const isPeakHour = hour >= 17 && hour <= 20;
  
  return {
    timestamp: `${hour.toString().padStart(2, '0')}:00`,
    chargingPower: isPeakHour ? 120 + Math.random() * 80 : 200 + Math.random() * 150,
    dischargingPower: isPeakHour ? 180 + Math.random() * 120 : 50 + Math.random() * 80,
    netFlow: isPeakHour ? -60 - Math.random() * 40 : 150 + Math.random() * 70,
    peakDemandOffset: isPeakHour ? 180 + Math.random() * 120 : 0,
    totalVehicles: isPeakHour ? 85 + Math.floor(Math.random() * 20) : 45 + Math.floor(Math.random() * 30)
  };
});

const BATTERY_COLORS = ['#3b82f6', '#10b981', '#f59e0b', '#ef4444'];

export default function ChargingAnalytics() {
  // V2G Session Limits - Business Rules
  const v2gLimits: V2GSessionLimits = {
    userLimit: 5,           // Max 5 sessions per individual
    manufacturerLimit: 12,  // Max 12 sessions allowed by car makers
    kwhPerSession: 50       // Each session supplies 50 KWh
  };

  const totalBatteries = mockBatteryAnalytics.reduce((sum, battery) => sum + battery.count, 0);
  const weightedAvgSOC = mockBatteryAnalytics.reduce((sum, battery) => 
    sum + (battery.avgSOC * battery.count), 0) / totalBatteries;
  const totalCapacity = mockBatteryAnalytics.reduce((sum, battery) => 
    sum + (battery.avgCapacity * battery.count), 0);
  const fastChargingCount = mockBatteryAnalytics.reduce((sum, battery) => 
    sum + (battery.fastChargingCapable ? battery.count : 0), 0);

  const currentSession = mockChargingSessions[mockChargingSessions.length - 1];
  const peakOffset = mockChargingSessions.reduce((sum, session) => sum + session.peakDemandOffset, 0);
  
  // Calculate max possible V2G capacity based on session limits and KWh per session
  const maxUserV2GCapacity = v2gLimits.userLimit * v2gLimits.kwhPerSession; // 5 * 50 = 250 KWh
  const maxManufacturerV2GCapacity = v2gLimits.manufacturerLimit * v2gLimits.kwhPerSession; // 12 * 50 = 600 KWh

  return (
    <div className="p-6 space-y-6" data-testid="page-charging-analytics">
      {/* Header */}
      <div className="space-y-2">
        <h1 className="text-3xl font-bold text-foreground">Charging & Battery Analytics</h1>
        <p className="text-muted-foreground">
          Monitor battery performance, charging patterns, and grid discharge analytics
        </p>
      </div>

      {/* V2G Session Limits Information */}
      <div className="grid grid-cols-1 md:grid-cols-3 gap-4 p-4 bg-card rounded-lg border">
        <div className="space-y-1">
          <p className="text-sm text-muted-foreground">Max Sessions per User</p>
          <p className="text-2xl font-bold">{v2gLimits.userLimit}</p>
          <p className="text-xs text-muted-foreground">Max capacity: {maxUserV2GCapacity} KWh</p>
        </div>
        <div className="space-y-1">
          <p className="text-sm text-muted-foreground">Max Sessions per Car Maker</p>
          <p className="text-2xl font-bold">{v2gLimits.manufacturerLimit}</p>
          <p className="text-xs text-muted-foreground">Max capacity: {maxManufacturerV2GCapacity} KWh</p>
        </div>
        <div className="space-y-1">
          <p className="text-sm text-muted-foreground">Energy per Session</p>
          <p className="text-2xl font-bold">{v2gLimits.kwhPerSession} KWh</p>
          <p className="text-xs text-muted-foreground">Standard V2G session size</p>
        </div>
      </div>

      {/* Overview Stats */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
        <Card data-testid="card-avg-soc">
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Fleet Avg SOC</CardTitle>
            <Battery className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">{weightedAvgSOC.toFixed(1)}%</div>
            <Progress value={weightedAvgSOC} className="h-2 mt-2" />
            <p className="text-xs text-muted-foreground mt-1">
              Across {totalBatteries} vehicles
            </p>
          </CardContent>
        </Card>

        <Card data-testid="card-total-capacity">
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Total Capacity</CardTitle>
            <Gauge className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">{(totalCapacity / 1000).toFixed(1)} MWh</div>
            <p className="text-xs text-muted-foreground">
              Fleet storage capacity
            </p>
          </CardContent>
        </Card>

        <Card data-testid="card-fast-charging">
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Fast Charging</CardTitle>
            <Zap className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-green-600">{fastChargingCount}</div>
            <p className="text-xs text-muted-foreground">
              {((fastChargingCount / totalBatteries) * 100).toFixed(1)}% of fleet
            </p>
          </CardContent>
        </Card>

        <Card data-testid="card-peak-offset">
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Peak Offset (24h)</CardTitle>
            <TrendingDown className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-blue-600">{(peakOffset / 1000).toFixed(1)} MWh</div>
            <p className="text-xs text-muted-foreground">
              Grid demand reduced
            </p>
          </CardContent>
        </Card>
      </div>

      {/* Battery Type Distribution */}
      <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
        <Card data-testid="card-battery-distribution">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <BarChart3 className="h-5 w-5" />
              Battery Type Distribution
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="h-64">
              <ResponsiveContainer width="100%" height="100%">
                <PieChart>
                  <Pie
                    data={mockBatteryAnalytics}
                    cx="50%"
                    cy="50%"
                    outerRadius={80}
                    dataKey="count"
                    nameKey="batteryType"
                    label={({ batteryType, count }) => `${batteryType.split(' ')[0]} (${count})`}
                  >
                    {mockBatteryAnalytics.map((_, index) => (
                      <Cell key={`cell-${index}`} fill={BATTERY_COLORS[index]} />
                    ))}
                  </Pie>
                  <Tooltip />
                </PieChart>
              </ResponsiveContainer>
            </div>
          </CardContent>
        </Card>

        <Card data-testid="card-charging-flow">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Activity className="h-5 w-5" />
              Charging/Discharging Flow (24h)
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="h-64">
              <ResponsiveContainer width="100%" height="100%">
                <LineChart data={mockChargingSessions}>
                  <CartesianGrid strokeDasharray="3 3" />
                  <XAxis dataKey="timestamp" />
                  <YAxis />
                  <Tooltip 
                    formatter={(value: number, name: string) => [
                      `${value.toFixed(1)} kW`, 
                      name === 'chargingPower' ? 'Grid → Vehicles' : 
                      name === 'dischargingPower' ? 'Vehicles → Grid' : 
                      'Net Flow'
                    ]}
                  />
                  <Line 
                    type="monotone" 
                    dataKey="chargingPower" 
                    stroke="#10b981" 
                    name="chargingPower"
                    strokeWidth={2}
                  />
                  <Line 
                    type="monotone" 
                    dataKey="dischargingPower" 
                    stroke="#3b82f6" 
                    name="dischargingPower"
                    strokeWidth={2}
                  />
                  <Line 
                    type="monotone" 
                    dataKey="netFlow" 
                    stroke="#f59e0b" 
                    name="netFlow"
                    strokeWidth={2}
                    strokeDasharray="5 5"
                  />
                </LineChart>
              </ResponsiveContainer>
            </div>
          </CardContent>
        </Card>
      </div>

      {/* Battery Analytics Details */}
      <Card data-testid="card-battery-details">
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Battery className="h-5 w-5" />
            Battery Technology Analysis
          </CardTitle>
        </CardHeader>
        <CardContent>
          <div className="space-y-4">
            {mockBatteryAnalytics.map((battery, index) => (
              <div 
                key={battery.batteryType} 
                className="border rounded-lg p-4 hover-elevate"
                data-testid={`battery-${index}`}
              >
                <div className="flex justify-between items-start mb-3">
                  <div>
                    <h3 className="font-semibold text-lg">{battery.batteryType}</h3>
                    <div className="text-sm text-muted-foreground">
                      {battery.count} vehicles • {battery.avgCapacity} kWh avg capacity
                    </div>
                  </div>
                  <div className="flex items-center gap-2">
                    {battery.fastChargingCapable && (
                      <Badge className="bg-green-500 text-white">
                        <Zap className="h-3 w-3 mr-1" />
                        Fast Charging
                      </Badge>
                    )}
                    <Badge className={`${BATTERY_COLORS[index]} text-white`} style={{backgroundColor: BATTERY_COLORS[index]}}>
                      {((battery.count / totalBatteries) * 100).toFixed(1)}%
                    </Badge>
                  </div>
                </div>

                <div className="grid grid-cols-1 md:grid-cols-5 gap-4">
                  <div className="space-y-2">
                    <div className="text-sm font-medium">Average SOC</div>
                    <div className="text-xl font-bold">{battery.avgSOC}%</div>
                    <Progress value={battery.avgSOC} className="h-2" />
                  </div>

                  <div className="space-y-2">
                    <div className="text-sm font-medium">Max Discharge</div>
                    <div className="text-xl font-bold text-blue-600">{battery.maxDischargeRate} kW</div>
                    <div className="text-xs text-muted-foreground">Per vehicle</div>
                  </div>

                  <div className="space-y-2">
                    <div className="text-sm font-medium">Cycle Count</div>
                    <div className="text-xl font-bold">{battery.cycleCount.toLocaleString()}</div>
                    <div className="text-xs text-muted-foreground">Average cycles</div>
                  </div>

                  <div className="space-y-2">
                    <div className="text-sm font-medium">Health Score</div>
                    <div className="text-xl font-bold text-green-600">{battery.healthScore}%</div>
                    <Progress value={battery.healthScore} className="h-2" />
                  </div>

                  <div className="space-y-2">
                    <div className="text-sm font-medium">Total Power</div>
                    <div className="text-xl font-bold">{(battery.count * battery.maxDischargeRate).toFixed(1)} kW</div>
                    <div className="text-xs text-muted-foreground">Fleet contribution</div>
                  </div>
                </div>
              </div>
            ))}
          </div>
        </CardContent>
      </Card>

      {/* Peak Discharge Performance */}
      <Card data-testid="card-peak-performance">
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <TrendingUp className="h-5 w-5" />
            Peak Discharge Performance
          </CardTitle>
        </CardHeader>
        <CardContent>
          <div className="h-64">
            <ResponsiveContainer width="100%" height="100%">
              <BarChart data={mockChargingSessions.filter((_, i) => i >= 17 && i <= 20)}>
                <CartesianGrid strokeDasharray="3 3" />
                <XAxis dataKey="timestamp" />
                <YAxis />
                <Tooltip 
                  formatter={(value: number, name: string) => [
                    `${value.toFixed(1)} ${name === 'totalVehicles' ? 'vehicles' : 'kW'}`, 
                    name === 'peakDemandOffset' ? 'Grid Support' : 'Active Vehicles'
                  ]}
                />
                <Bar dataKey="peakDemandOffset" fill="#3b82f6" name="peakDemandOffset" />
                <Bar dataKey="totalVehicles" fill="#10b981" name="totalVehicles" />
              </BarChart>
            </ResponsiveContainer>
          </div>
          <div className="mt-4 grid grid-cols-1 md:grid-cols-3 gap-4 text-center">
            <div>
              <div className="text-2xl font-bold text-blue-600">{currentSession.dischargingPower.toFixed(1)} kW</div>
              <div className="text-sm text-muted-foreground">Current Discharge Rate</div>
            </div>
            <div>
              <div className="text-2xl font-bold text-green-600">{currentSession.totalVehicles}</div>
              <div className="text-sm text-muted-foreground">Active Vehicles</div>
            </div>
            <div>
              <div className="text-2xl font-bold text-orange-600">{currentSession.peakDemandOffset.toFixed(1)} kW</div>
              <div className="text-sm text-muted-foreground">Peak Offset Provided</div>
            </div>
          </div>
        </CardContent>
      </Card>
    </div>
  );
}