import { useState } from "react";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Slider } from "@/components/ui/slider";
import { Switch } from "@/components/ui/switch";
import { Label } from "@/components/ui/label";
import { Separator } from "@/components/ui/separator";
import { Alert, AlertDescription } from "@/components/ui/alert";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import VehicleMap, { Vehicle } from "@/components/VehicleMap";
import { motion } from "framer-motion";
import { 
  Search,
  Settings2,
  Battery,
  MapPin,
  Clock,
  Zap,
  Car,
  Send,
  Filter,
  Users,
  Shield,
  AlertTriangle,
  Play,
  Sparkles,
  Target,
  TrendingUp,
  DollarSign,
  Gauge,
  Brain,
  Lightbulb
} from "lucide-react";

// Extended mock data for Singapore simulation
const singaporeVehicles: Vehicle[] = [
  // Central Region
  { id: 101, name: "EV-CBD-001", lat: 1.2847, lng: 103.8506, soc: 85, status: "Idle", battery: 75, fleet: false },
  { id: 102, name: "EV-CBD-002", lat: 1.2966, lng: 103.8520, soc: 65, status: "Charging", battery: 60, fleet: true },
  { id: 103, name: "EV-CBD-003", lat: 1.2785, lng: 103.8478, soc: 90, status: "Idle", battery: 80, fleet: false },
  
  // North Region
  { id: 201, name: "EV-YST-001", lat: 1.4390, lng: 103.7869, soc: 75, status: "Idle", battery: 65, fleet: false },
  { id: 202, name: "EV-YST-002", lat: 1.4270, lng: 103.8350, soc: 45, status: "Offline", battery: 50, fleet: true },
  { id: 203, name: "EV-SBW-001", lat: 1.3521, lng: 103.8198, soc: 80, status: "Discharging", battery: 70, fleet: false },
  
  // East Region  
  { id: 301, name: "EV-TPY-001", lat: 1.3621, lng: 103.8950, soc: 70, status: "Idle", battery: 75, fleet: false },
  { id: 302, name: "EV-BED-001", lat: 1.3240, lng: 103.9280, soc: 55, status: "Charging", battery: 55, fleet: true },
  { id: 303, name: "EV-PSR-001", lat: 1.3721, lng: 103.9450, soc: 85, status: "Idle", battery: 80, fleet: false },
  
  // West Region
  { id: 401, name: "EV-JUR-001", lat: 1.3390, lng: 103.7050, soc: 60, status: "Idle", battery: 65, fleet: false },
  { id: 402, name: "EV-CLE-001", lat: 1.3160, lng: 103.7640, soc: 80, status: "Idle", battery: 75, fleet: true },
  { id: 403, name: "EV-BST-001", lat: 1.3500, lng: 103.7400, soc: 40, status: "Charging", battery: 50, fleet: false },
  
  // South Region
  { id: 501, name: "EV-HGB-001", lat: 1.2700, lng: 103.8200, soc: 90, status: "Idle", battery: 85, fleet: false },
  { id: 502, name: "EV-SEN-001", lat: 1.2330, lng: 103.8350, soc: 75, status: "Discharging", battery: 70, fleet: true },
  { id: 503, name: "EV-GEY-001", lat: 1.2800, lng: 103.8600, soc: 65, status: "Idle", battery: 60, fleet: false }
];

interface CriteriaSettings {
  minSoc: number;
  maxSoc: number;
  minBattery: number;
  maxBattery: number;
  includeFleet: boolean;
  includePrivate: boolean;
  allowCharging: boolean;
  allowDischarging: boolean;
  allowIdle: boolean;
  allowOffline: boolean;
  minIdleTime: number; // in hours
  prioritizeLocation: boolean;
}

export default function CriteriaManagement() {
  const [criteria, setCriteria] = useState<CriteriaSettings>({
    minSoc: 20,
    maxSoc: 100,
    minBattery: 40,
    maxBattery: 100,
    includeFleet: true,
    includePrivate: true,
    allowCharging: false,
    allowDischarging: false,
    allowIdle: true,
    allowOffline: false,
    minIdleTime: 2,
    prioritizeLocation: true
  });
  
  const [simulationRunning, setSimulationRunning] = useState(false);
  const [selectedVehicle, setSelectedVehicle] = useState<Vehicle | null>(null);
  const [filteredVehicles, setFilteredVehicles] = useState<Vehicle[]>([]);

  const handleSimulate = () => {
    setSimulationRunning(true);
    
    // Apply criteria filters
    const filtered = singaporeVehicles.filter(vehicle => {
      // SoC criteria
      if (vehicle.soc < criteria.minSoc || vehicle.soc > criteria.maxSoc) return false;
      
      // Battery capacity criteria
      if (vehicle.battery < criteria.minBattery || vehicle.battery > criteria.maxBattery) return false;
      
      // Fleet/Private criteria
      if (!criteria.includeFleet && vehicle.fleet) return false;
      if (!criteria.includePrivate && !vehicle.fleet) return false;
      
      // Status criteria
      const statusCheck = {
        "Charging": criteria.allowCharging,
        "Discharging": criteria.allowDischarging,
        "Idle": criteria.allowIdle,
        "Offline": criteria.allowOffline
      }[vehicle.status];
      
      if (!statusCheck) return false;
      
      return true;
    });
    
    setFilteredVehicles(filtered);
    
    setTimeout(() => {
      setSimulationRunning(false);
    }, 2000);
  };

  const handleVehicleClick = (vehicle: Vehicle) => {
    setSelectedVehicle(vehicle);
  };

  const handleTriggerPing = () => {
    if (selectedVehicle) {
      // Create ping notification for selected vehicle
      console.log(`Triggering ping for vehicle: ${selectedVehicle.name}`);
      
      // TODO: Call API to create ping notification
      // const pingData = {
      //   title: `V2G Request for ${selectedVehicle.name}`,
      //   message: `Grid needs support. Discharge available: ${selectedVehicle.soc}% SoC`,
      //   userId: 'demo-user', // Replace with actual user
      //   vehicleId: selectedVehicle.id.toString(),
      //   type: 'grid_request'
      // };
      
      alert(`Ping sent to ${selectedVehicle.name} for V2G participation!`);
    }
  };

  const eligibleCount = filteredVehicles.length;
  const totalCapacity = filteredVehicles.reduce((sum, v) => sum + v.battery, 0);
  const avgSoc = filteredVehicles.length > 0 
    ? Math.round(filteredVehicles.reduce((sum, v) => sum + v.soc, 0) / filteredVehicles.length) 
    : 0;

  return (
    <div className="min-h-screen bg-background p-4 md:p-6">
      <div className="max-w-7xl mx-auto space-y-6">
        {/* Header */}
        <div className="space-y-2">
          <h1 className="text-3xl font-bold flex items-center gap-3">
            <Settings2 className="h-8 w-8 text-primary" />
            V2G Selection Criteria
          </h1>
          <p className="text-muted-foreground">
            Define criteria to select eligible vehicles for V2G participation across Singapore
          </p>
        </div>

        <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
          {/* Criteria Configuration Panel */}
          <div className="lg:col-span-1 space-y-4">
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <Filter className="h-5 w-5" />
                  Selection Criteria
                </CardTitle>
              </CardHeader>
              <CardContent className="space-y-6">
                {/* SoC Range */}
                <div className="space-y-3">
                  <Label className="text-sm font-medium">State of Charge Range</Label>
                  <div className="space-y-2">
                    <div className="flex justify-between text-xs text-muted-foreground">
                      <span>{criteria.minSoc}%</span>
                      <span>{criteria.maxSoc}%</span>
                    </div>
                    <div className="px-2">
                      <Slider
                        value={[criteria.minSoc, criteria.maxSoc]}
                        onValueChange={([min, max]) => setCriteria(prev => ({ ...prev, minSoc: min, maxSoc: max }))}
                        max={100}
                        min={0}
                        step={5}
                        className="w-full"
                      />
                    </div>
                  </div>
                </div>

                {/* Battery Capacity Range */}
                <div className="space-y-3">
                  <Label className="text-sm font-medium">Battery Capacity Range (kWh)</Label>
                  <div className="space-y-2">
                    <div className="flex justify-between text-xs text-muted-foreground">
                      <span>{criteria.minBattery}</span>
                      <span>{criteria.maxBattery}</span>
                    </div>
                    <div className="px-2">
                      <Slider
                        value={[criteria.minBattery, criteria.maxBattery]}
                        onValueChange={([min, max]) => setCriteria(prev => ({ ...prev, minBattery: min, maxBattery: max }))}
                        max={150}
                        min={30}
                        step={5}
                        className="w-full"
                      />
                    </div>
                  </div>
                </div>

                <Separator />

                {/* Vehicle Type */}
                <div className="space-y-3">
                  <Label className="text-sm font-medium">Vehicle Types</Label>
                  <div className="space-y-2">
                    <div className="flex items-center justify-between">
                      <div className="flex items-center gap-2">
                        <Users className="h-4 w-4" />
                        <span className="text-sm">Fleet Vehicles</span>
                      </div>
                      <Switch
                        checked={criteria.includeFleet}
                        onCheckedChange={(checked) => setCriteria(prev => ({ ...prev, includeFleet: checked }))}
                      />
                    </div>
                    <div className="flex items-center justify-between">
                      <div className="flex items-center gap-2">
                        <Car className="h-4 w-4" />
                        <span className="text-sm">Private EVs</span>
                      </div>
                      <Switch
                        checked={criteria.includePrivate}
                        onCheckedChange={(checked) => setCriteria(prev => ({ ...prev, includePrivate: checked }))}
                      />
                    </div>
                  </div>
                </div>

                <Separator />

                {/* Status Filters */}
                <div className="space-y-3">
                  <Label className="text-sm font-medium">Allowed Status</Label>
                  <div className="space-y-2">
                    <div className="flex items-center justify-between">
                      <span className="text-sm">Idle</span>
                      <Switch
                        checked={criteria.allowIdle}
                        onCheckedChange={(checked) => setCriteria(prev => ({ ...prev, allowIdle: checked }))}
                      />
                    </div>
                    <div className="flex items-center justify-between">
                      <span className="text-sm">Charging</span>
                      <Switch
                        checked={criteria.allowCharging}
                        onCheckedChange={(checked) => setCriteria(prev => ({ ...prev, allowCharging: checked }))}
                      />
                    </div>
                    <div className="flex items-center justify-between">
                      <span className="text-sm">Discharging</span>
                      <Switch
                        checked={criteria.allowDischarging}
                        onCheckedChange={(checked) => setCriteria(prev => ({ ...prev, allowDischarging: checked }))}
                      />
                    </div>
                  </div>
                </div>

                <Separator />

                {/* Advanced Options */}
                <div className="space-y-3">
                  <div className="flex items-center justify-between">
                    <div className="flex items-center gap-2">
                      <MapPin className="h-4 w-4" />
                      <span className="text-sm">Prioritize by Location</span>
                    </div>
                    <Switch
                      checked={criteria.prioritizeLocation}
                      onCheckedChange={(checked) => setCriteria(prev => ({ ...prev, prioritizeLocation: checked }))}
                    />
                  </div>
                </div>

                {/* Simulate Button */}
                <Button 
                  onClick={handleSimulate}
                  className="w-full gap-2"
                  disabled={simulationRunning}
                >
                  {simulationRunning ? (
                    <>
                      <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-current" />
                      Simulating...
                    </>
                  ) : (
                    <>
                      <Play className="h-4 w-4" />
                      Simulate Singapore Grid
                    </>
                  )}
                </Button>
              </CardContent>
            </Card>

            {/* Results Summary */}
            {filteredVehicles.length > 0 && (
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <Search className="h-5 w-5" />
                    Simulation Results
                  </CardTitle>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div className="grid grid-cols-2 gap-4 text-center">
                    <div>
                      <div className="text-2xl font-bold text-primary">{eligibleCount}</div>
                      <div className="text-sm text-muted-foreground">Eligible Vehicles</div>
                    </div>
                    <div>
                      <div className="text-2xl font-bold text-green-600">{totalCapacity}</div>
                      <div className="text-sm text-muted-foreground">Total kWh</div>
                    </div>
                  </div>
                  
                  <div className="text-center">
                    <div className="text-xl font-semibold">{avgSoc}%</div>
                    <div className="text-sm text-muted-foreground">Average SoC</div>
                  </div>

                  {selectedVehicle && (
                    <div className="mt-4 p-3 bg-primary/5 rounded-lg">
                      <div className="text-sm font-medium mb-2">Selected Vehicle:</div>
                      <div className="space-y-1">
                        <div className="font-semibold">{selectedVehicle.name}</div>
                        <div className="text-sm text-muted-foreground">
                          SoC: {selectedVehicle.soc}% • Battery: {selectedVehicle.battery}kWh
                        </div>
                        <Badge className="text-xs">{selectedVehicle.status}</Badge>
                      </div>
                      
                      <Button 
                        onClick={handleTriggerPing}
                        size="sm"
                        className="w-full mt-3 gap-2"
                      >
                        <Send className="h-3 w-3" />
                        Trigger V2G Ping
                      </Button>
                    </div>
                  )}
                </CardContent>
              </Card>
            )}
          </div>

          {/* Singapore Map */}
          <div className="lg:col-span-2">
            <Card className="h-[700px]">
              <CardHeader>
                <CardTitle className="flex items-center justify-between">
                  <div className="flex items-center gap-2">
                    <MapPin className="h-5 w-5" />
                    Singapore V2G Network
                  </div>
                  {filteredVehicles.length > 0 && (
                    <Badge variant="outline" className="gap-1">
                      <Car className="h-3 w-3" />
                      {filteredVehicles.length} eligible
                    </Badge>
                  )}
                </CardTitle>
              </CardHeader>
              <CardContent className="h-full p-0">
                <VehicleMap
                  vehicles={simulationRunning ? [] : (filteredVehicles.length > 0 ? filteredVehicles : singaporeVehicles)}
                  selectedVehicle={selectedVehicle}
                  onVehicleSelect={handleVehicleClick}
                  className="h-full w-full rounded-lg"
                  showScoreBubbles={false}
                  showHotZones={false}
                  allowHotZoneEditing={false}
                  showSocInfo={true}
                  v2gCalls={[]}
                  gridInvitations={[]}
                  vehicleScores={[]}
                  hotZones={[]}
                  onRegionSelect={() => {}}
                  onHotZoneSelect={() => {}}
                />
              </CardContent>
            </Card>
          </div>
        </div>

        {/* Intelligent Optimizer */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Brain className="h-5 w-5 text-primary" />
              Smart V2G Optimizer
            </CardTitle>
          </CardHeader>
          <CardContent>
            <Tabs defaultValue="intelligent" className="w-full">
              <TabsList className="grid w-full grid-cols-5 mb-6">
                <TabsTrigger value="intelligent" className="gap-1.5" data-testid="tab-intelligent">
                  <Sparkles className="h-4 w-4" />
                  AI Optimizer
                </TabsTrigger>
                <TabsTrigger value="capacity" className="gap-1.5" data-testid="tab-capacity">
                  <Battery className="h-4 w-4" />
                  Max Capacity
                </TabsTrigger>
                <TabsTrigger value="location" className="gap-1.5" data-testid="tab-location">
                  <MapPin className="h-4 w-4" />
                  Location-Based
                </TabsTrigger>
                <TabsTrigger value="cost" className="gap-1.5" data-testid="tab-cost">
                  <DollarSign className="h-4 w-4" />
                  Cost Effective
                </TabsTrigger>
                <TabsTrigger value="balanced" className="gap-1.5" data-testid="tab-balanced">
                  <Gauge className="h-4 w-4" />
                  Balanced
                </TabsTrigger>
              </TabsList>

              {/* AI Optimizer Tab */}
              <TabsContent value="intelligent" className="space-y-4">
                <Alert>
                  <Lightbulb className="h-4 w-4" />
                  <AlertDescription>
                    AI-driven optimization analyzes real-time grid conditions, historical performance, and predictive models to recommend optimal V2G participation criteria.
                  </AlertDescription>
                </Alert>
                
                <div className="grid md:grid-cols-3 gap-4">
                  <Card className="hover-elevate">
                    <CardHeader className="pb-3">
                      <CardTitle className="text-sm font-medium flex items-center gap-2">
                        <TrendingUp className="h-4 w-4 text-green-600" />
                        Predicted Participation
                      </CardTitle>
                    </CardHeader>
                    <CardContent>
                      <div className="text-2xl font-bold">87%</div>
                      <p className="text-xs text-muted-foreground mt-1">
                        Based on current criteria and historical acceptance rates
                      </p>
                    </CardContent>
                  </Card>

                  <Card className="hover-elevate">
                    <CardHeader className="pb-3">
                      <CardTitle className="text-sm font-medium flex items-center gap-2">
                        <Zap className="h-4 w-4 text-blue-600" />
                        Grid Impact Score
                      </CardTitle>
                    </CardHeader>
                    <CardContent>
                      <div className="text-2xl font-bold">9.2/10</div>
                      <p className="text-xs text-muted-foreground mt-1">
                        High effectiveness for peak demand offset
                      </p>
                    </CardContent>
                  </Card>

                  <Card className="hover-elevate">
                    <CardHeader className="pb-3">
                      <CardTitle className="text-sm font-medium flex items-center gap-2">
                        <Target className="h-4 w-4 text-purple-600" />
                        Optimization Level
                      </CardTitle>
                    </CardHeader>
                    <CardContent>
                      <div className="text-2xl font-bold">Excellent</div>
                      <p className="text-xs text-muted-foreground mt-1">
                        Criteria well-balanced for grid needs
                      </p>
                    </CardContent>
                  </Card>
                </div>

                <div className="space-y-3">
                  <h4 className="font-semibold text-sm flex items-center gap-2">
                    <Sparkles className="h-4 w-4" />
                    AI Recommendations
                  </h4>
                  <div className="space-y-2">
                    <div className="flex items-start gap-3 p-3 bg-primary/5 rounded-lg hover-elevate">
                      <div className="mt-0.5">
                        <Badge variant="outline" className="bg-green-500/10 text-green-700 border-green-200">
                          Optimize
                        </Badge>
                      </div>
                      <div className="flex-1 text-sm">
                        <p className="font-medium">Increase SOC minimum to 30%</p>
                        <p className="text-muted-foreground text-xs mt-0.5">
                          Ensures sufficient reserve capacity for extended discharge periods during peak hours
                        </p>
                      </div>
                      <Button size="sm" variant="outline" data-testid="button-apply-rec-1">Apply</Button>
                    </div>

                    <div className="flex items-start gap-3 p-3 bg-primary/5 rounded-lg hover-elevate">
                      <div className="mt-0.5">
                        <Badge variant="outline" className="bg-blue-500/10 text-blue-700 border-blue-200">
                          Enhance
                        </Badge>
                      </div>
                      <div className="flex-1 text-sm">
                        <p className="font-medium">Prioritize vehicles in Central & North regions</p>
                        <p className="text-muted-foreground text-xs mt-0.5">
                          Grid demand analysis shows 42% higher consumption in these areas during peak
                        </p>
                      </div>
                      <Button size="sm" variant="outline" data-testid="button-apply-rec-2">Apply</Button>
                    </div>

                    <div className="flex items-start gap-3 p-3 bg-primary/5 rounded-lg hover-elevate">
                      <div className="mt-0.5">
                        <Badge variant="outline" className="bg-yellow-500/10 text-yellow-700 border-yellow-200">
                          Consider
                        </Badge>
                      </div>
                      <div className="flex-1 text-sm">
                        <p className="font-medium">Include fleet vehicles with battery &gt; 70 kWh</p>
                        <p className="text-muted-foreground text-xs mt-0.5">
                          Fleet EVs show 94% reliability and provide significant grid support capacity
                        </p>
                      </div>
                      <Button size="sm" variant="outline" data-testid="button-apply-rec-3">Apply</Button>
                    </div>
                  </div>
                </div>
              </TabsContent>

              {/* Max Capacity Tab */}
              <TabsContent value="capacity" className="space-y-4">
                <Alert>
                  <Battery className="h-4 w-4" />
                  <AlertDescription>
                    Maximize total energy capacity available for grid support. Prioritizes vehicles with largest batteries and highest SOC.
                  </AlertDescription>
                </Alert>

                <div className="grid md:grid-cols-2 gap-4">
                  <div className="space-y-3">
                    <h4 className="font-semibold text-sm">Recommended Settings</h4>
                    <div className="space-y-2 text-sm">
                      <div className="flex justify-between p-2 bg-muted/50 rounded">
                        <span className="text-muted-foreground">Min SOC:</span>
                        <span className="font-medium">70%</span>
                      </div>
                      <div className="flex justify-between p-2 bg-muted/50 rounded">
                        <span className="text-muted-foreground">Min Battery:</span>
                        <span className="font-medium">65 kWh</span>
                      </div>
                      <div className="flex justify-between p-2 bg-muted/50 rounded">
                        <span className="text-muted-foreground">Vehicle Types:</span>
                        <span className="font-medium">Fleet + Private</span>
                      </div>
                      <div className="flex justify-between p-2 bg-muted/50 rounded">
                        <span className="text-muted-foreground">Status:</span>
                        <span className="font-medium">Idle Only</span>
                      </div>
                    </div>
                    <Button className="w-full gap-2" data-testid="button-apply-capacity">
                      <Target className="h-4 w-4" />
                      Apply Max Capacity Profile
                    </Button>
                  </div>

                  <div className="space-y-3">
                    <h4 className="font-semibold text-sm">Expected Outcomes</h4>
                    <div className="space-y-2">
                      <div className="p-3 border rounded-lg">
                        <div className="text-2xl font-bold text-primary">~850 kWh</div>
                        <p className="text-xs text-muted-foreground">Total Available Capacity</p>
                      </div>
                      <div className="p-3 border rounded-lg">
                        <div className="text-2xl font-bold text-green-600">8-12</div>
                        <p className="text-xs text-muted-foreground">Eligible Vehicles</p>
                      </div>
                      <div className="p-3 border rounded-lg">
                        <div className="text-2xl font-bold text-blue-600">~75%</div>
                        <p className="text-xs text-muted-foreground">Avg SOC Level</p>
                      </div>
                    </div>
                  </div>
                </div>
              </TabsContent>

              {/* Location-Based Tab */}
              <TabsContent value="location" className="space-y-4">
                <Alert>
                  <MapPin className="h-4 w-4" />
                  <AlertDescription>
                    Optimize based on proximity to high-demand areas and grid infrastructure. Minimizes transmission losses.
                  </AlertDescription>
                </Alert>

                <div className="space-y-3">
                  <h4 className="font-semibold text-sm">Priority Regions</h4>
                  <div className="grid md:grid-cols-2 gap-3">
                    {[
                      { region: "Central Business District", demand: "High", vehicles: 3, priority: "1" },
                      { region: "North (Yishun/Sembawang)", demand: "Medium", vehicles: 3, priority: "2" },
                      { region: "East (Tampines/Bedok)", demand: "Medium", vehicles: 3, priority: "3" },
                      { region: "West (Jurong/Clementi)", demand: "Low", vehicles: 3, priority: "4" }
                    ].map((area) => (
                      <div key={area.region} className="p-3 border rounded-lg hover-elevate">
                        <div className="flex items-start justify-between mb-2">
                          <div className="flex-1">
                            <div className="font-medium text-sm">{area.region}</div>
                            <div className="text-xs text-muted-foreground">{area.vehicles} vehicles available</div>
                          </div>
                          <Badge variant={area.demand === "High" ? "default" : "secondary"} className="text-xs">
                            {area.demand}
                          </Badge>
                        </div>
                        <div className="flex items-center gap-2">
                          <span className="text-xs text-muted-foreground">Priority:</span>
                          <Badge variant="outline" className="text-xs">#{area.priority}</Badge>
                        </div>
                      </div>
                    ))}
                  </div>
                  <Button className="w-full gap-2" data-testid="button-apply-location">
                    <MapPin className="h-4 w-4" />
                    Apply Location-Based Profile
                  </Button>
                </div>
              </TabsContent>

              {/* Cost Effective Tab */}
              <TabsContent value="cost" className="space-y-4">
                <Alert>
                  <DollarSign className="h-4 w-4" />
                  <AlertDescription>
                    Minimize grid operational costs while maintaining service quality. Balance incentives with grid needs.
                  </AlertDescription>
                </Alert>

                <div className="grid md:grid-cols-2 gap-4">
                  <div className="space-y-3">
                    <h4 className="font-semibold text-sm">Cost Optimization Strategy</h4>
                    <div className="space-y-2 text-sm">
                      <div className="p-3 bg-muted/50 rounded-lg">
                        <div className="flex items-center justify-between mb-1">
                          <span className="font-medium">Lower Incentive Tier</span>
                          <Badge variant="outline" className="text-xs">$0.45/kWh</Badge>
                        </div>
                        <p className="text-xs text-muted-foreground">Target high-SOC idle vehicles (less compensation needed)</p>
                      </div>
                      <div className="p-3 bg-muted/50 rounded-lg">
                        <div className="flex items-center justify-between mb-1">
                          <span className="font-medium">Reduce Peak Demand</span>
                          <Badge variant="outline" className="text-xs">Save 15%</Badge>
                        </div>
                        <p className="text-xs text-muted-foreground">Strategic discharge during expensive peak hours</p>
                      </div>
                      <div className="p-3 bg-muted/50 rounded-lg">
                        <div className="flex items-center justify-between mb-1">
                          <span className="font-medium">Fleet Partnership</span>
                          <Badge variant="outline" className="text-xs">Bulk Rate</Badge>
                        </div>
                        <p className="text-xs text-muted-foreground">Negotiated rates with commercial fleet operators</p>
                      </div>
                    </div>
                  </div>

                  <div className="space-y-3">
                    <h4 className="font-semibold text-sm">Cost Savings Projection</h4>
                    <div className="space-y-2">
                      <div className="p-3 border rounded-lg bg-green-50 dark:bg-green-950/20">
                        <div className="text-2xl font-bold text-green-600">$2,450</div>
                        <p className="text-xs text-muted-foreground">Monthly Grid Cost Reduction</p>
                      </div>
                      <div className="p-3 border rounded-lg">
                        <div className="text-xl font-bold">$0.38/kWh</div>
                        <p className="text-xs text-muted-foreground">Effective Cost (vs $0.52 peak rate)</p>
                      </div>
                      <div className="p-3 border rounded-lg">
                        <div className="text-xl font-bold">27%</div>
                        <p className="text-xs text-muted-foreground">Cost Efficiency Improvement</p>
                      </div>
                    </div>
                    <Button className="w-full gap-2" data-testid="button-apply-cost">
                      <DollarSign className="h-4 w-4" />
                      Apply Cost-Effective Profile
                    </Button>
                  </div>
                </div>
              </TabsContent>

              {/* Balanced Tab */}
              <TabsContent value="balanced" className="space-y-4">
                <Alert>
                  <Gauge className="h-4 w-4" />
                  <AlertDescription>
                    Balanced approach optimizing across capacity, location, cost, and user satisfaction. Recommended for most scenarios.
                  </AlertDescription>
                </Alert>

                <div className="space-y-4">
                  <div className="grid md:grid-cols-4 gap-3">
                    <div className="text-center p-3 border rounded-lg">
                      <Battery className="h-5 w-5 mx-auto mb-2 text-blue-600" />
                      <div className="text-lg font-bold">Good</div>
                      <p className="text-xs text-muted-foreground">Capacity</p>
                    </div>
                    <div className="text-center p-3 border rounded-lg">
                      <MapPin className="h-5 w-5 mx-auto mb-2 text-green-600" />
                      <div className="text-lg font-bold">Optimal</div>
                      <p className="text-xs text-muted-foreground">Location</p>
                    </div>
                    <div className="text-center p-3 border rounded-lg">
                      <DollarSign className="h-5 w-5 mx-auto mb-2 text-yellow-600" />
                      <div className="text-lg font-bold">Fair</div>
                      <p className="text-xs text-muted-foreground">Cost</p>
                    </div>
                    <div className="text-center p-3 border rounded-lg">
                      <Users className="h-5 w-5 mx-auto mb-2 text-purple-600" />
                      <div className="text-lg font-bold">High</div>
                      <p className="text-xs text-muted-foreground">User Satisfaction</p>
                    </div>
                  </div>

                  <div className="grid md:grid-cols-2 gap-4">
                    <div className="space-y-2">
                      <h4 className="font-semibold text-sm">Balanced Criteria</h4>
                      <div className="space-y-1.5 text-sm">
                        <div className="flex justify-between p-2 bg-muted/50 rounded">
                          <span>SOC Range:</span>
                          <span className="font-medium">40-90%</span>
                        </div>
                        <div className="flex justify-between p-2 bg-muted/50 rounded">
                          <span>Battery Range:</span>
                          <span className="font-medium">50-100 kWh</span>
                        </div>
                        <div className="flex justify-between p-2 bg-muted/50 rounded">
                          <span>Location Priority:</span>
                          <span className="font-medium">Medium</span>
                        </div>
                        <div className="flex justify-between p-2 bg-muted/50 rounded">
                          <span>Mixed Fleet:</span>
                          <span className="font-medium">Yes</span>
                        </div>
                      </div>
                    </div>

                    <div className="space-y-2">
                      <h4 className="font-semibold text-sm">Expected Results</h4>
                      <div className="space-y-1.5">
                        <div className="p-3 border rounded-lg">
                          <div className="text-xl font-bold">10-14</div>
                          <p className="text-xs text-muted-foreground">Eligible Vehicles</p>
                        </div>
                        <div className="p-3 border rounded-lg">
                          <div className="text-xl font-bold">~650 kWh</div>
                          <p className="text-xs text-muted-foreground">Total Capacity</p>
                        </div>
                        <div className="p-3 border rounded-lg">
                          <div className="text-xl font-bold">85%</div>
                          <p className="text-xs text-muted-foreground">Participation Rate</p>
                        </div>
                      </div>
                    </div>
                  </div>

                  <Button className="w-full gap-2" data-testid="button-apply-balanced">
                    <Gauge className="h-4 w-4" />
                    Apply Balanced Profile
                  </Button>
                </div>
              </TabsContent>
            </Tabs>
          </CardContent>
        </Card>
      </div>
    </div>
  );
}