import { useState } from "react";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { Progress } from "@/components/ui/progress";
import { 
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { 
  DollarSign, 
  Zap, 
  Clock, 
  MapPin, 
  TrendingUp, 
  AlertTriangle,
  Target,
  Battery
} from "lucide-react";

interface GridRequirement {
  id: string;
  region: string;
  peakDemand: number; // MW
  currentSupply: number; // MW
  offsetNeeded: number; // MW
  duration: number; // minutes
  startTime: string;
  endTime: string;
  costPerMWh: number;
  incentivePerDischarge: number;
  priority: "critical" | "high" | "medium" | "low";
  location: {
    lat: number;
    lng: number;
    address: string;
  };
}

// Mock grid requirements data - Total Peak Demand: 7GW, Total Offset: 50MW
const mockGridRequirements: GridRequirement[] = [
  {
    id: "req-1",
    region: "Central Business District",
    peakDemand: 500,
    currentSupply: 495.1,
    offsetNeeded: 4.9,
    duration: 90,
    startTime: "18:00",
    endTime: "19:30",
    costPerMWh: 250,
    incentivePerDischarge: 100,
    priority: "critical",
    location: {
      lat: 1.3521,
      lng: 103.8198,
      address: "Marina Bay Financial Centre"
    }
  },
  {
    id: "req-2", 
    region: "Orchard Shopping District",
    peakDemand: 750,
    currentSupply: 745,
    offsetNeeded: 5.0,
    duration: 60,
    startTime: "19:00",
    endTime: "20:00",
    costPerMWh: 180,
    incentivePerDischarge: 75,
    priority: "high",
    location: {
      lat: 1.3048,
      lng: 103.8318,
      address: "Orchard Road Commercial Area"
    }
  },
  {
    id: "req-3",
    region: "Jurong Industrial Estate",
    peakDemand: 1300,
    currentSupply: 1295,
    offsetNeeded: 5.0,
    duration: 45,
    startTime: "17:30",
    endTime: "18:15",
    costPerMWh: 160,
    incentivePerDischarge: 65,
    priority: "medium",
    location: {
      lat: 1.3294,
      lng: 103.7414,
      address: "Jurong Industrial Park"
    }
  },
  {
    id: "req-4",
    region: "Woodlands North",
    peakDemand: 600,
    currentSupply: 595,
    offsetNeeded: 5.0,
    duration: 75,
    startTime: "18:30",
    endTime: "19:45",
    costPerMWh: 145,
    incentivePerDischarge: 60,
    priority: "medium",
    location: {
      lat: 1.4390,
      lng: 103.7869,
      address: "Woodlands Industrial Zone"
    }
  },
  {
    id: "req-5",
    region: "Tampines East",
    peakDemand: 600,
    currentSupply: 594.9,
    offsetNeeded: 5.1,
    duration: 60,
    startTime: "18:00",
    endTime: "19:00",
    costPerMWh: 155,
    incentivePerDischarge: 68,
    priority: "high",
    location: {
      lat: 1.3621,
      lng: 103.8950,
      address: "Tampines Regional Centre"
    }
  },
  {
    id: "req-6",
    region: "Changi Airport Cluster",
    peakDemand: 1250,
    currentSupply: 1245,
    offsetNeeded: 5.0,
    duration: 120,
    startTime: "17:00",
    endTime: "19:00",
    costPerMWh: 220,
    incentivePerDischarge: 95,
    priority: "critical",
    location: {
      lat: 1.3644,
      lng: 103.9915,
      address: "Changi Airport Area"
    }
  },
  {
    id: "req-7",
    region: "Tuas West",
    peakDemand: 500,
    currentSupply: 495,
    offsetNeeded: 5.0,
    duration: 50,
    startTime: "18:15",
    endTime: "19:05",
    costPerMWh: 150,
    incentivePerDischarge: 62,
    priority: "low",
    location: {
      lat: 1.3400,
      lng: 103.6360,
      address: "Tuas Industrial Area"
    }
  },
  {
    id: "req-8",
    region: "Punggol Digital District",
    peakDemand: 500,
    currentSupply: 495,
    offsetNeeded: 5.0,
    duration: 80,
    startTime: "18:45",
    endTime: "20:05",
    costPerMWh: 175,
    incentivePerDischarge: 72,
    priority: "medium",
    location: {
      lat: 1.4043,
      lng: 103.9024,
      address: "Punggol Way"
    }
  },
  {
    id: "req-9",
    region: "Sentosa Resort",
    peakDemand: 800,
    currentSupply: 795,
    offsetNeeded: 5.0,
    duration: 70,
    startTime: "19:30",
    endTime: "20:40",
    costPerMWh: 200,
    incentivePerDischarge: 85,
    priority: "high",
    location: {
      lat: 1.2494,
      lng: 103.8303,
      address: "Sentosa Island"
    }
  },
  {
    id: "req-10",
    region: "Seletar Aerospace",
    peakDemand: 200,
    currentSupply: 195,
    offsetNeeded: 5.0,
    duration: 55,
    startTime: "17:45",
    endTime: "18:40",
    costPerMWh: 165,
    incentivePerDischarge: 70,
    priority: "low",
    location: {
      lat: 1.4167,
      lng: 103.8691,
      address: "Seletar Aerospace Park"
    }
  }
];

const getPriorityColor = (priority: string) => {
  switch (priority) {
    case "critical": return "bg-red-500";
    case "high": return "bg-orange-500";
    case "medium": return "bg-yellow-500";
    case "low": return "bg-green-500";
    default: return "bg-gray-500";
  }
};

const getPriorityTextColor = (priority: string) => {
  switch (priority) {
    case "critical": return "text-red-600";
    case "high": return "text-orange-600";
    case "medium": return "text-yellow-600";
    case "low": return "text-green-600";
    default: return "text-gray-600";
  }
};

export default function GridRequirements() {
  const [selectedRegion, setSelectedRegion] = useState<string>("all");

  // Filter requirements based on selected region
  const filteredRequirements = selectedRegion === "all" 
    ? mockGridRequirements 
    : mockGridRequirements.filter(req => req.region === selectedRegion);

  const totalPeakDemand = filteredRequirements.reduce((sum, req) => sum + req.peakDemand, 0);
  const totalOffsetNeeded = filteredRequirements.reduce((sum, req) => sum + req.offsetNeeded, 0);
  const totalCost = filteredRequirements.reduce((sum, req) => sum + (req.offsetNeeded * req.costPerMWh), 0);
  const criticalRequirements = filteredRequirements.filter(req => req.priority === "critical").length;

  // Get unique regions for dropdown
  const regions = ["all", ...mockGridRequirements.map(req => req.region)];

  return (
    <div className="p-6 space-y-6" data-testid="page-grid-requirements">
      {/* Header */}
      <div className="flex flex-col md:flex-row md:items-center md:justify-between gap-4">
        <div className="space-y-2">
          <h1 className="text-3xl font-bold text-foreground">Grid Requirements Dashboard</h1>
          <p className="text-muted-foreground">
            Monitor peak demand forecasts, offset requirements, and grid balancing costs
          </p>
        </div>
        <div className="w-full md:w-64">
          <Select value={selectedRegion} onValueChange={setSelectedRegion}>
            <SelectTrigger data-testid="select-region">
              <SelectValue placeholder="Select region" />
            </SelectTrigger>
            <SelectContent>
              <SelectItem value="all">All Regions</SelectItem>
              {mockGridRequirements.map(req => (
                <SelectItem key={req.id} value={req.region}>
                  {req.region}
                </SelectItem>
              ))}
            </SelectContent>
          </Select>
        </div>
      </div>

      {/* Overview Stats */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
        <Card data-testid="card-total-peak-demand">
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Total Peak Demand</CardTitle>
            <Zap className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">
              {totalPeakDemand >= 1000 ? `${(totalPeakDemand / 1000).toFixed(1)} GW` : `${totalPeakDemand} MW`}
            </div>
            <p className="text-xs text-muted-foreground">
              Across {selectedRegion === "all" ? "all regions" : "selected region"}
            </p>
          </CardContent>
        </Card>

        <Card data-testid="card-offset-needed">
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Offset Needed</CardTitle>
            <Target className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-red-600">{totalOffsetNeeded} MW</div>
            <p className="text-xs text-muted-foreground">
              V2G discharge required
            </p>
          </CardContent>
        </Card>

        <Card data-testid="card-total-cost">
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Total Cost</CardTitle>
            <DollarSign className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">${(totalCost / 1000).toFixed(1)}k</div>
            <p className="text-xs text-muted-foreground">
              Grid balancing budget
            </p>
          </CardContent>
        </Card>

        <Card data-testid="card-critical-areas">
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Critical Areas</CardTitle>
            <AlertTriangle className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-red-600">{criticalRequirements}</div>
            <p className="text-xs text-muted-foreground">
              Urgent intervention needed
            </p>
          </CardContent>
        </Card>
      </div>

      {/* Grid Requirements List */}
      <Card data-testid="card-requirements-list">
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <TrendingUp className="h-5 w-5" />
            Regional Peak Demand Requirements
          </CardTitle>
        </CardHeader>
        <CardContent>
          <div className="space-y-4">
            {filteredRequirements.map((requirement) => (
              <div 
                key={requirement.id} 
                className="border rounded-lg p-4 hover-elevate"
                data-testid={`requirement-${requirement.id}`}
              >
                <div className="flex justify-between items-start mb-3">
                  <div className="space-y-1">
                    <h3 className="font-semibold text-lg">{requirement.region}</h3>
                    <div className="flex items-center gap-2 text-sm text-muted-foreground">
                      <MapPin className="h-4 w-4" />
                      <span>{requirement.location.address}</span>
                    </div>
                  </div>
                  <Badge 
                    className={`${getPriorityColor(requirement.priority)} text-white`}
                    data-testid={`priority-${requirement.priority}`}
                  >
                    {requirement.priority.toUpperCase()}
                  </Badge>
                </div>

                <div className="grid grid-cols-1 md:grid-cols-3 gap-4 mb-4">
                  <div className="space-y-2">
                    <div className="text-sm font-medium">Peak Demand</div>
                    <div className="text-2xl font-bold">{requirement.peakDemand} MW</div>
                    <div className="text-xs text-muted-foreground">Current: {requirement.currentSupply} MW</div>
                  </div>

                  <div className="space-y-2">
                    <div className="text-sm font-medium">Offset Required</div>
                    <div className={`text-2xl font-bold ${getPriorityTextColor(requirement.priority)}`}>
                      {requirement.offsetNeeded} MW
                    </div>
                    <div className="text-xs text-muted-foreground">
                      {((requirement.offsetNeeded / requirement.peakDemand) * 100).toFixed(1)}% reduction needed
                    </div>
                  </div>

                  <div className="space-y-2">
                    <div className="text-sm font-medium">Duration</div>
                    <div className="text-2xl font-bold">{requirement.duration} min</div>
                    <div className="text-xs text-muted-foreground">
                      {requirement.startTime} - {requirement.endTime}
                    </div>
                  </div>
                </div>

                <div className="mb-4">
                  <div className="flex justify-between text-sm mb-1">
                    <span>Supply vs Demand</span>
                    <span>{((requirement.currentSupply / requirement.peakDemand) * 100).toFixed(1)}%</span>
                  </div>
                  <Progress 
                    value={(requirement.currentSupply / requirement.peakDemand) * 100} 
                    className="h-2"
                  />
                </div>

                <div className="flex justify-between items-center">
                  <div className="space-y-1">
                    <div className="text-sm text-muted-foreground">Incentive Rate</div>
                    <div className="text-lg font-bold text-green-600">
                      ${requirement.incentivePerDischarge}/discharge
                    </div>
                  </div>
                  
                  <div className="space-y-1">
                    <div className="text-sm text-muted-foreground">Grid Cost</div>
                    <div className="text-lg font-bold">
                      ${requirement.costPerMWh}/MWh
                    </div>
                  </div>

                  <Button 
                    variant="default" 
                    className="ml-4"
                    data-testid={`dispatch-${requirement.id}`}
                  >
                    <Battery className="h-4 w-4 mr-2" />
                    Dispatch V2G
                  </Button>
                </div>
              </div>
            ))}
          </div>
        </CardContent>
      </Card>
    </div>
  );
}