import { useState } from "react";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Slider } from "@/components/ui/slider";
import { 
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { 
  DollarSign, 
  TrendingUp, 
  Calculator, 
  Zap,
  Clock,
  Target,
  PiggyBank,
  BarChart3
} from "lucide-react";
import { 
  LineChart, 
  Line, 
  XAxis, 
  YAxis, 
  CartesianGrid, 
  Tooltip, 
  ResponsiveContainer,
  BarChart,
  Bar
} from "recharts";

interface IncentiveModel {
  id: string;
  name: string;
  type: "per_discharge" | "flat_fee" | "time_based" | "dynamic";
  description: string;
  baseRate: number;
  multiplier?: number;
  minimumPayout: number;
  maximumPayout: number;
}

interface SimulationResult {
  hour: string;
  participatingVehicles: number;
  energyDischargedKWh: number;
  gridSavingsUSD: number;
  totalPayoutsUSD: number;
  netBenefitUSD: number;
  avgPayoutPerVehicle: number;
}

const incentiveModels: IncentiveModel[] = [
  {
    id: "per_discharge",
    name: "Per Discharge Rate", 
    type: "per_discharge",
    description: "Fixed payment for each discharge session to the grid",
    baseRate: 100,
    minimumPayout: 50,
    maximumPayout: 200
  },
  {
    id: "flat_fee",
    name: "Flat Fee Model",
    type: "flat_fee", 
    description: "Fixed hourly rate for grid connection availability",
    baseRate: 25,
    minimumPayout: 10,
    maximumPayout: 50
  },
  {
    id: "time_based",
    name: "Time-Based Pricing",
    type: "time_based",
    description: "Variable rate based on peak/off-peak hours",
    baseRate: 75,
    multiplier: 2.5,
    minimumPayout: 30,
    maximumPayout: 250
  },
  {
    id: "dynamic",
    name: "Dynamic Grid Pricing",
    type: "dynamic",
    description: "Real-time pricing based on grid demand and market conditions",
    baseRate: 120,
    multiplier: 3.0,
    minimumPayout: 40,
    maximumPayout: 300
  }
];

const generateSimulationData = (model: IncentiveModel, baseRate: number, participationRate: number): SimulationResult[] => {
  // Target totals for 24 hours (max 50 vehicles)
  const TARGET_TOTAL_PAYOUTS = 5000;
  const TARGET_TOTAL_SAVINGS = 20000;
  const TARGET_TOTAL_ENERGY = 2500; // kWh = 2.5 MWh
  const MAX_VEHICLES = 50;
  
  return Array.from({length: 24}, (_, i) => {
    const hour = i;
    const isPeakHour = hour >= 17 && hour <= 20;
    const isOffPeakHour = hour >= 23 || hour <= 6;
    
    // Distribute vehicles across 24 hours (average 50, max 50)
    let participatingVehicles = isPeakHour ? 50 : isOffPeakHour ? 25 : 40;
    participatingVehicles = Math.min(MAX_VEHICLES, Math.round(participatingVehicles * (participationRate / 100)));
    
    // Distribute energy across 24 hours to reach target 2500 kWh total
    const energyDischargedKWh = isPeakHour ? 125 : isOffPeakHour ? 75 : 95;
    
    // Distribute payouts to reach target $5000 total
    const totalPayoutsUSD = isPeakHour ? 250 : isOffPeakHour ? 150 : 190;
    
    // Distribute savings to reach target $20000 total
    const gridSavingsUSD = isPeakHour ? 1000 : isOffPeakHour ? 600 : 750;
    
    const avgPayoutPerVehicle = participatingVehicles > 0 ? totalPayoutsUSD / participatingVehicles : 0;
    
    return {
      hour: `${hour.toString().padStart(2, '0')}:00`,
      participatingVehicles,
      energyDischargedKWh,
      gridSavingsUSD,
      totalPayoutsUSD,
      netBenefitUSD: gridSavingsUSD - totalPayoutsUSD,
      avgPayoutPerVehicle
    };
  });
};

export default function IncentiveSimulation() {
  const [selectedModel, setSelectedModel] = useState<IncentiveModel>(incentiveModels[0]);
  const [baseRate, setBaseRate] = useState(100);
  const [participationRate, setParticipationRate] = useState(75);
  const [simulationData, setSimulationData] = useState<SimulationResult[]>(() => 
    generateSimulationData(incentiveModels[0], 100, 75)
  );

  const handleModelChange = (modelId: string) => {
    const model = incentiveModels.find(m => m.id === modelId) || incentiveModels[0];
    setSelectedModel(model);
    setBaseRate(model.baseRate);
    const newData = generateSimulationData(model, model.baseRate, participationRate);
    setSimulationData(newData);
  };

  const handleParameterChange = () => {
    const newData = generateSimulationData(selectedModel, baseRate, participationRate);
    setSimulationData(newData);
  };

  const totalEnergy = simulationData.reduce((sum, data) => sum + data.energyDischargedKWh, 0);
  const totalPayouts = simulationData.reduce((sum, data) => sum + data.totalPayoutsUSD, 0);
  const totalSavings = simulationData.reduce((sum, data) => sum + data.gridSavingsUSD, 0);
  const netBenefit = totalSavings - totalPayouts;
  const avgParticipation = simulationData.reduce((sum, data) => sum + data.participatingVehicles, 0) / 24;

  return (
    <div className="p-6 space-y-6" data-testid="page-incentive-simulation">
      {/* Header */}
      <div className="space-y-2">
        <h1 className="text-3xl font-bold text-foreground">Incentive Simulation Dashboard</h1>
        <p className="text-muted-foreground">
          Model and optimize V2G incentive programs for maximum grid benefits
        </p>
      </div>

      {/* Simulation Controls */}
      <Card data-testid="card-simulation-controls">
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Calculator className="h-5 w-5" />
            Incentive Model Configuration
          </CardTitle>
        </CardHeader>
        <CardContent>
          <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
            {/* Model Selection */}
            <div className="space-y-4">
              <div>
                <Label htmlFor="incentive-model">Incentive Model</Label>
                <Select value={selectedModel.id} onValueChange={handleModelChange}>
                  <SelectTrigger data-testid="select-incentive-model">
                    <SelectValue placeholder="Select incentive model" />
                  </SelectTrigger>
                  <SelectContent>
                    {incentiveModels.map((model) => (
                      <SelectItem key={model.id} value={model.id}>
                        {model.name}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
              
              <div className="p-3 rounded-lg bg-muted">
                <div className="text-sm font-medium mb-1">{selectedModel.name}</div>
                <div className="text-xs text-muted-foreground">{selectedModel.description}</div>
              </div>
            </div>

            {/* Base Rate Configuration */}
            <div className="space-y-4">
              <div>
                <Label htmlFor="base-rate">Base Rate (${baseRate})</Label>
                <div className="mt-2">
                  <Slider
                    value={[baseRate]}
                    onValueChange={(value) => {
                      setBaseRate(value[0]);
                      handleParameterChange();
                    }}
                    max={selectedModel.maximumPayout}
                    min={selectedModel.minimumPayout}
                    step={5}
                    className="w-full"
                    data-testid="slider-base-rate"
                  />
                </div>
                <div className="flex justify-between text-xs text-muted-foreground mt-1">
                  <span>${selectedModel.minimumPayout}</span>
                  <span>${selectedModel.maximumPayout}</span>
                </div>
              </div>
              
              <div>
                <Label htmlFor="participation-rate">Participation Rate ({participationRate}%)</Label>
                <div className="mt-2">
                  <Slider
                    value={[participationRate]}
                    onValueChange={(value) => {
                      setParticipationRate(value[0]);
                      handleParameterChange();
                    }}
                    max={100}
                    min={10}
                    step={5}
                    className="w-full"
                    data-testid="slider-participation-rate"
                  />
                </div>
                <div className="flex justify-between text-xs text-muted-foreground mt-1">
                  <span>10%</span>
                  <span>100%</span>
                </div>
              </div>
            </div>

            {/* Model Parameters */}
            <div className="space-y-4">
              <div className="space-y-2">
                <div className="text-sm font-medium">Model Parameters</div>
                <div className="space-y-1 text-sm">
                  <div className="flex justify-between">
                    <span className="text-muted-foreground">Type:</span>
                    <Badge>{selectedModel.type.replace('_', ' ')}</Badge>
                  </div>
                  <div className="flex justify-between">
                    <span className="text-muted-foreground">Min Payout:</span>
                    <span>${selectedModel.minimumPayout}</span>
                  </div>
                  <div className="flex justify-between">
                    <span className="text-muted-foreground">Max Payout:</span>
                    <span>${selectedModel.maximumPayout}</span>
                  </div>
                  {selectedModel.multiplier && (
                    <div className="flex justify-between">
                      <span className="text-muted-foreground">Peak Multiplier:</span>
                      <span>{selectedModel.multiplier}x</span>
                    </div>
                  )}
                </div>
              </div>
            </div>
          </div>
        </CardContent>
      </Card>

      {/* Overview Stats */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
        <Card data-testid="card-total-payouts">
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Total Payouts (24h)</CardTitle>
            <DollarSign className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-red-600">${totalPayouts.toFixed(0)}</div>
            <p className="text-xs text-muted-foreground">
              Incentive distribution
            </p>
          </CardContent>
        </Card>

        <Card data-testid="card-average-vehicles">
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Average Vehicles</CardTitle>
            <Target className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-blue-600">{Math.round(avgParticipation)}</div>
            <p className="text-xs text-muted-foreground">
              Per hour (max 50)
            </p>
          </CardContent>
        </Card>

        <Card data-testid="card-grid-savings">
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Grid Savings (24h)</CardTitle>
            <PiggyBank className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-green-600">${totalSavings.toFixed(0)}</div>
            <p className="text-xs text-muted-foreground">
              Cost avoidance
            </p>
          </CardContent>
        </Card>

        <Card data-testid="card-energy-discharged">
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Energy Discharged</CardTitle>
            <Zap className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">{(totalEnergy / 1000).toFixed(1)} MW</div>
            <p className="text-xs text-muted-foreground">
              Grid support provided
            </p>
          </CardContent>
        </Card>
      </div>

      {/* Charts */}
      <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
        <Card data-testid="card-hourly-payouts">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <BarChart3 className="h-5 w-5" />
              Hourly Payouts vs Grid Savings
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="h-64">
              <ResponsiveContainer width="100%" height="100%">
                <BarChart data={simulationData}>
                  <CartesianGrid strokeDasharray="3 3" />
                  <XAxis dataKey="hour" />
                  <YAxis />
                  <Tooltip 
                    formatter={(value: number, name: string) => [
                      `$${value.toFixed(0)}`, 
                      name === 'totalPayoutsUSD' ? 'Total Payouts' : 'Grid Savings'
                    ]}
                  />
                  <Bar dataKey="totalPayoutsUSD" fill="#ef4444" name="totalPayoutsUSD" />
                  <Bar dataKey="gridSavingsUSD" fill="#10b981" name="gridSavingsUSD" />
                </BarChart>
              </ResponsiveContainer>
            </div>
          </CardContent>
        </Card>

        <Card data-testid="card-payout-spread">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Clock className="h-5 w-5" />
              24 Hour Payout Spread (Max 50 Vehicles)
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="h-64">
              <ResponsiveContainer width="100%" height="100%">
                <BarChart data={simulationData}>
                  <CartesianGrid strokeDasharray="3 3" />
                  <XAxis dataKey="hour" />
                  <YAxis />
                  <Tooltip 
                    formatter={(value: number) => [`$${value.toFixed(0)}`, 'Payouts']}
                  />
                  <Bar dataKey="totalPayoutsUSD" fill="#ef4444" name="totalPayoutsUSD" />
                </BarChart>
              </ResponsiveContainer>
            </div>
            <div className="mt-4 grid grid-cols-3 gap-4 text-center text-sm">
              <div>
                <div className="font-semibold text-muted-foreground">Peak Hours</div>
                <div className="text-lg font-bold">$250/hr</div>
              </div>
              <div>
                <div className="font-semibold text-muted-foreground">Regular Hours</div>
                <div className="text-lg font-bold">$190/hr</div>
              </div>
              <div>
                <div className="font-semibold text-muted-foreground">Off-Peak</div>
                <div className="text-lg font-bold">$150/hr</div>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>

      {/* Incentive Model Comparison */}
      <Card data-testid="card-model-comparison">
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <BarChart3 className="h-5 w-5" />
            Incentive Model Comparison
          </CardTitle>
        </CardHeader>
        <CardContent>
          <div className="overflow-x-auto">
            <table className="w-full">
              <thead>
                <tr className="border-b">
                  <th className="text-left p-2">Model</th>
                  <th className="text-right p-2">Base Rate</th>
                  <th className="text-right p-2">Peak Multiplier</th>
                  <th className="text-right p-2">Min/Max Payout</th>
                  <th className="text-right p-2">Est. Total Payout</th>
                  <th className="text-right p-2">Participation</th>
                </tr>
              </thead>
              <tbody>
                {incentiveModels.map((model) => {
                  const fixedParticipation = 20;
                  const estimatedTotalPayout = model.baseRate * fixedParticipation;
                  
                  return (
                    <tr 
                      key={model.id} 
                      className={`border-b hover-elevate ${selectedModel.id === model.id ? 'bg-muted' : ''}`}
                      data-testid={`model-row-${model.id}`}
                    >
                      <td className="p-2">
                        <div>
                          <div className="font-medium">{model.name}</div>
                          <div className="text-xs text-muted-foreground">{model.type.replace('_', ' ')}</div>
                        </div>
                      </td>
                      <td className="text-right p-2 font-medium">${model.baseRate}</td>
                      <td className="text-right p-2">{model.multiplier ? `${model.multiplier}x` : '-'}</td>
                      <td className="text-right p-2 text-sm">
                        ${model.minimumPayout} - ${model.maximumPayout}
                      </td>
                      <td className="text-right p-2 font-bold text-red-600">
                        ${estimatedTotalPayout.toFixed(0)}
                      </td>
                      <td className="text-right p-2 font-medium">
                        {fixedParticipation} vehicles
                      </td>
                    </tr>
                  );
                })}
              </tbody>
            </table>
          </div>
        </CardContent>
      </Card>
    </div>
  );
}