import { useState } from "react";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Switch } from "@/components/ui/switch";
import { Label } from "@/components/ui/label";
import { Separator } from "@/components/ui/separator";
import { Badge } from "@/components/ui/badge";
import { useNotifications } from "@/hooks/useNotifications";
import { useTheme } from "@/components/ThemeProvider";
import { 
  Bell, 
  Smartphone, 
  Moon, 
  Sun, 
  Volume2, 
  VolumeX,
  Shield,
  Info
} from "lucide-react";

export default function Settings() {
  const { 
    isSupported, 
    isSubscribed, 
    permission, 
    subscribe, 
    unsubscribe, 
    requestPermission 
  } = useNotifications();
  const { theme, setTheme } = useTheme();
  
  const [soundEnabled, setSoundEnabled] = useState(true);
  const [vibrationEnabled, setVibrationEnabled] = useState(true);

  const handleNotificationToggle = async (enabled: boolean) => {
    if (enabled && !isSubscribed) {
      try {
        await subscribe();
      } catch (error) {
        console.error('Failed to enable notifications:', error);
      }
    } else if (!enabled && isSubscribed) {
      try {
        await unsubscribe();
      } catch (error) {
        console.error('Failed to disable notifications:', error);
      }
    }
  };

  const getNotificationStatus = () => {
    if (!isSupported) return { status: 'Not Supported', color: 'destructive' };
    if (permission === 'denied') return { status: 'Blocked', color: 'destructive' };
    if (permission === 'default') return { status: 'Not Requested', color: 'secondary' };
    if (isSubscribed) return { status: 'Active', color: 'default' };
    return { status: 'Inactive', color: 'secondary' };
  };

  const notificationStatus = getNotificationStatus();

  return (
    <div className="min-h-screen bg-background p-4 md:p-6">
      <div className="max-w-2xl mx-auto space-y-6">
        <div className="space-y-2">
          <h1 className="text-3xl font-bold" data-testid="title-settings">V2G Settings</h1>
          <p className="text-muted-foreground">
            Manage your V2G platform preferences and notifications
          </p>
        </div>

        {/* Notifications Settings */}
        <Card data-testid="card-notifications">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Bell className="h-5 w-5" />
              Push Notifications
            </CardTitle>
          </CardHeader>
          <CardContent className="space-y-4">
            <div className="flex items-center justify-between">
              <div className="flex items-center gap-3">
                <div className="space-y-1">
                  <Label className="text-sm font-medium">Enable Notifications</Label>
                  <p className="text-xs text-muted-foreground">
                    Receive real-time V2G alerts and grid invitations
                  </p>
                </div>
              </div>
              <div className="flex items-center gap-2">
                <Badge 
                  variant={notificationStatus.color as any}
                  className="text-xs"
                  data-testid={`badge-status-${notificationStatus.status.toLowerCase().replace(' ', '-')}`}
                >
                  {notificationStatus.status}
                </Badge>
                <Switch
                  checked={isSubscribed}
                  onCheckedChange={handleNotificationToggle}
                  disabled={!isSupported || permission === 'denied'}
                  data-testid="switch-notifications"
                />
              </div>
            </div>

            {permission === 'denied' && (
              <div className="flex items-start gap-2 p-3 bg-destructive/10 border border-destructive/20 rounded-lg">
                <Shield className="h-4 w-4 text-destructive mt-0.5" />
                <div className="text-xs">
                  <p className="font-medium text-destructive">Notifications Blocked</p>
                  <p className="text-muted-foreground">
                    Enable in your browser settings to receive V2G alerts
                  </p>
                </div>
              </div>
            )}

            <Separator />

            <div className="space-y-3">
              <div className="flex items-center justify-between">
                <div className="flex items-center gap-2">
                  <Volume2 className="h-4 w-4" />
                  <Label className="text-sm">Sound Alerts</Label>
                </div>
                <Switch
                  checked={soundEnabled}
                  onCheckedChange={setSoundEnabled}
                  disabled={!isSubscribed}
                  data-testid="switch-sound"
                />
              </div>

              <div className="flex items-center justify-between">
                <div className="flex items-center gap-2">
                  <Smartphone className="h-4 w-4" />
                  <Label className="text-sm">Vibration</Label>
                </div>
                <Switch
                  checked={vibrationEnabled}
                  onCheckedChange={setVibrationEnabled}
                  disabled={!isSubscribed}
                  data-testid="switch-vibration"
                />
              </div>
            </div>
          </CardContent>
        </Card>

        {/* Appearance Settings */}
        <Card data-testid="card-appearance">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              {theme === 'dark' ? <Moon className="h-5 w-5" /> : <Sun className="h-5 w-5" />}
              Appearance
            </CardTitle>
          </CardHeader>
          <CardContent className="space-y-4">
            <div className="flex items-center justify-between">
              <div className="space-y-1">
                <Label className="text-sm font-medium">Theme</Label>
                <p className="text-xs text-muted-foreground">
                  Choose your preferred color scheme
                </p>
              </div>
              <div className="flex gap-2">
                <Button
                  variant={theme === 'light' ? 'default' : 'outline'}
                  size="sm"
                  onClick={() => setTheme('light')}
                  className="gap-2"
                  data-testid="button-theme-light"
                >
                  <Sun className="h-3 w-3" />
                  Light
                </Button>
                <Button
                  variant={theme === 'dark' ? 'default' : 'outline'}
                  size="sm"
                  onClick={() => setTheme('dark')}
                  className="gap-2"
                  data-testid="button-theme-dark"
                >
                  <Moon className="h-3 w-3" />
                  Dark
                </Button>
                <Button
                  variant={theme === 'system' ? 'default' : 'outline'}
                  size="sm"
                  onClick={() => setTheme('system')}
                  className="gap-2"
                  data-testid="button-theme-system"
                >
                  <Smartphone className="h-3 w-3" />
                  Auto
                </Button>
              </div>
            </div>
          </CardContent>
        </Card>

        {/* System Information */}
        <Card data-testid="card-system-info">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Info className="h-5 w-5" />
              System Information
            </CardTitle>
          </CardHeader>
          <CardContent className="space-y-3">
            <div className="grid grid-cols-2 gap-4 text-sm">
              <div>
                <p className="font-medium">Notification Support</p>
                <p className="text-muted-foreground">
                  {isSupported ? 'Supported' : 'Not Supported'}
                </p>
              </div>
              <div>
                <p className="font-medium">Permission Status</p>
                <p className="text-muted-foreground capitalize">{permission}</p>
              </div>
              <div>
                <p className="font-medium">Service Worker</p>
                <p className="text-muted-foreground">
                  {navigator.serviceWorker ? 'Available' : 'Not Available'}
                </p>
              </div>
              <div>
                <p className="font-medium">Platform</p>
                <p className="text-muted-foreground">
                  {/Mobile|Android|iPhone/i.test(navigator.userAgent) ? 'Mobile' : 'Desktop'}
                </p>
              </div>
            </div>
          </CardContent>
        </Card>

        {/* Actions */}
        <div className="flex gap-2 pt-2">
          {permission !== 'granted' && isSupported && (
            <Button
              onClick={requestPermission}
              variant="outline"
              className="gap-2"
              data-testid="button-request-permission"
            >
              <Bell className="h-4 w-4" />
              Request Notification Permission
            </Button>
          )}
        </div>
      </div>
    </div>
  );
}