import type { Express } from "express";
import { createServer, type Server } from "http";
import { storage } from "./storage";
import { 
  insertNotificationSubscriptionSchema, 
  insertPingSchema,
  insertV2GSessionSchema,
} from "@shared/schema";
import { z } from "zod";
import { setupAuth } from "./auth";

export async function registerRoutes(app: Express): Promise<Server> {
  // sets up /api/register, /api/login, /api/logout, /api/user
  // from blueprint:javascript_auth_all_persistance
  setupAuth(app);

  // put application routes here
  // prefix all routes with /api

  // use storage to perform CRUD operations on the storage interface
  // e.g. storage.insertUser(user) or storage.getUserByUsername(username)

  app.get("/api/health", (req, res) => {
    res.json({ status: "ok" });
  });

  // Notification subscription routes
  app.post("/api/notifications/subscribe", async (req, res) => {
    try {
      const validatedData = insertNotificationSubscriptionSchema.parse(req.body);
      const subscription = await storage.createNotificationSubscription(validatedData);
      res.json({ success: true, subscription });
    } catch (error) {
      if (error instanceof z.ZodError) {
        res.status(400).json({ error: "Invalid subscription data", details: error.errors });
      } else {
        console.error("Error creating subscription:", error);
        res.status(500).json({ error: "Failed to create subscription" });
      }
    }
  });

  app.delete("/api/notifications/subscribe/:userId", async (req, res) => {
    try {
      const { userId } = req.params;
      const deleted = await storage.deleteNotificationSubscription(userId);
      if (deleted) {
        res.json({ success: true, message: "Subscription deleted" });
      } else {
        res.status(404).json({ error: "Subscription not found" });
      }
    } catch (error) {
      console.error("Error deleting subscription:", error);
      res.status(500).json({ error: "Failed to delete subscription" });
    }
  });

  app.get("/api/notifications/subscriptions", async (req, res) => {
    try {
      const subscriptions = await storage.getAllNotificationSubscriptions();
      res.json({ subscriptions });
    } catch (error) {
      console.error("Error fetching subscriptions:", error);
      res.status(500).json({ error: "Failed to fetch subscriptions" });
    }
  });

  // Ping notification routes
  app.post("/api/pings", async (req, res) => {
    try {
      const validatedData = insertPingSchema.parse(req.body);
      const ping = await storage.createPing(validatedData);
      
      // Broadcast to WebSocket clients for real-time delivery
      if (global.wsClients) {
        const message = JSON.stringify({
          type: 'PING_CREATED',
          data: ping
        });
        global.wsClients.forEach((ws: any) => {
          if (ws.readyState === 1) { // WebSocket.OPEN
            ws.send(message);
          }
        });
      }
      
      // Send Web Push notifications to subscribed users  
      try {
        const subscriptions = await storage.getAllNotificationSubscriptions();
        const targetSubscriptions = subscriptions.filter(sub => 
          !ping.userId || sub.userId === ping.userId
        );
        
        for (const subscription of targetSubscriptions) {
          // In production, use web-push library to send actual push notifications
          console.log(`[Push] Would send to ${subscription.userId}: ${ping.title}`);
        }
      } catch (pushError) {
        console.warn('Failed to send push notifications:', pushError);
      }
      
      res.json({ success: true, ping });
    } catch (error) {
      if (error instanceof z.ZodError) {
        res.status(400).json({ error: "Invalid ping data", details: error.errors });
      } else {
        console.error("Error creating ping:", error);
        res.status(500).json({ error: "Failed to create ping" });
      }
    }
  });

  app.get("/api/pings/user/:userId", async (req, res) => {
    try {
      const { userId } = req.params;
      const pings = await storage.getPingsByUserId(userId);
      res.json({ pings });
    } catch (error) {
      console.error("Error fetching user pings:", error);
      res.status(500).json({ error: "Failed to fetch pings" });
    }
  });

  app.get("/api/pings/region/:region", async (req, res) => {
    try {
      const { region } = req.params;
      const pings = await storage.getPingsByRegion(region);
      res.json({ pings });
    } catch (error) {
      console.error("Error fetching region pings:", error);
      res.status(500).json({ error: "Failed to fetch pings" });
    }
  });

  app.get("/api/pings/recent", async (req, res) => {
    try {
      const hoursAgo = parseInt(req.query.hours as string) || 24;
      const since = new Date(Date.now() - hoursAgo * 60 * 60 * 1000);
      const pings = await storage.getRecentPings(since);
      res.json({ pings });
    } catch (error) {
      console.error("Error fetching recent pings:", error);
      res.status(500).json({ error: "Failed to fetch pings" });
    }
  });

  // V2G Session routes - Max 5 per user, Max 12 per manufacturer
  app.post("/api/v2g/sessions", async (req, res) => {
    try {
      const validatedData = insertV2GSessionSchema.parse(req.body);
      
      // Check user session limit (max 5)
      const userSessionCount = await storage.getUserV2GSessionCount(validatedData.userId);
      if (userSessionCount >= 5) {
        return res.status(400).json({ 
          error: "Session limit reached", 
          message: "Maximum 5 active V2G sessions allowed per user",
          currentCount: userSessionCount,
          limit: 5
        });
      }
      
      // Check manufacturer session limit (max 12)
      const manufacturerSessionCount = await storage.getManufacturerV2GSessionCount(validatedData.manufacturer);
      if (manufacturerSessionCount >= 12) {
        return res.status(400).json({ 
          error: "Car maker limit reached", 
          message: `Maximum 12 active V2G sessions allowed for ${validatedData.manufacturer}`,
          currentCount: manufacturerSessionCount,
          limit: 12
        });
      }
      
      const session = await storage.createV2GSession(validatedData);
      res.json({ success: true, session });
    } catch (error) {
      if (error instanceof z.ZodError) {
        res.status(400).json({ error: "Invalid session data", details: error.errors });
      } else {
        console.error("Error creating V2G session:", error);
        res.status(500).json({ error: "Failed to create V2G session" });
      }
    }
  });

  app.get("/api/v2g/sessions/user/:userId", async (req, res) => {
    try {
      const { userId } = req.params;
      const sessions = await storage.getActiveV2GSessionsByUser(userId);
      const count = await storage.getUserV2GSessionCount(userId);
      res.json({ sessions, count, limit: 5 });
    } catch (error) {
      console.error("Error fetching user V2G sessions:", error);
      res.status(500).json({ error: "Failed to fetch V2G sessions" });
    }
  });

  app.get("/api/v2g/sessions/manufacturer/:manufacturer", async (req, res) => {
    try {
      const { manufacturer } = req.params;
      const sessions = await storage.getActiveV2GSessionsByManufacturer(manufacturer);
      const count = await storage.getManufacturerV2GSessionCount(manufacturer);
      res.json({ sessions, count, limit: 12 });
    } catch (error) {
      console.error("Error fetching manufacturer V2G sessions:", error);
      res.status(500).json({ error: "Failed to fetch V2G sessions" });
    }
  });

  app.patch("/api/v2g/sessions/:sessionId/complete", async (req, res) => {
    try {
      const { sessionId } = req.params;
      const { incentiveEarned } = req.body;
      const endTime = new Date().toISOString();
      
      const session = await storage.completeV2GSession(
        sessionId, 
        endTime, 
        incentiveEarned || 0
      );
      res.json({ success: true, session });
    } catch (error) {
      console.error("Error completing V2G session:", error);
      res.status(500).json({ error: "Failed to complete V2G session" });
    }
  });

  const httpServer = createServer(app);

  return httpServer;
}
