import { sql } from "drizzle-orm";
import { pgTable, text, varchar, integer, real, boolean } from "drizzle-orm/pg-core";
import { createInsertSchema } from "drizzle-zod";
import { z } from "zod";

export const users = pgTable("users", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  username: text("username").notNull().unique(),
  password: text("password").notNull(),
});

export const vehicles = pgTable("vehicles", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  name: text("name").notNull(),
  manufacturer: text("manufacturer").notNull(), // Car maker: Tesla, Nissan, BYD, etc.
  userId: varchar("user_id"), // References users.id - owner of the vehicle
  lat: real("lat").notNull(),
  lng: real("lng").notNull(),
  address: text("address"), // Location address
  district: text("district"), // Location district/area
  soc: integer("soc").notNull(), // State of Charge percentage
  socFloor: integer("soc_floor").default(20), // Minimum SOC user wants to maintain
  status: text("status").notNull(), // Charging, Discharging, Idle, Offline
  battery: integer("battery").notNull(), // Battery capacity in kWh
  maxDischargeRate: integer("max_discharge_rate"), // Maximum discharge rate in kW
  chargeCycles: integer("charge_cycles").default(0), // Number of charge cycles
  v2gEnrolled: boolean("v2g_enrolled").default(false), // V2G program enrollment
  fleet: boolean("fleet").notNull().default(false),
  lastSeen: text("last_seen"), // Last seen timestamp
});

export const v2gCalls = pgTable("v2g_calls", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  lat: real("lat").notNull(),
  lng: real("lng").notNull(),
  timestamp: text("timestamp").notNull(),
  magnitude: integer("magnitude").notNull(), // Call intensity/importance level
  region: text("region"), // Grid cell or region identifier
});

export const insertUserSchema = createInsertSchema(users).pick({
  username: true,
  password: true,
});

export const insertVehicleSchema = createInsertSchema(vehicles).omit({
  id: true,
});

export const insertV2GCallSchema = createInsertSchema(v2gCalls).omit({
  id: true,
});

export const gridInvitations = pgTable("grid_invitations", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  lat: real("lat").notNull(),
  lng: real("lng").notNull(),
  timestamp: text("timestamp").notNull(),
  demandLevel: text("demand_level").notNull(), // "peak", "high", "medium"
  incentiveAmount: real("incentive_amount"), // Payment offered for connection
  priority: integer("priority").notNull(), // 1-10 priority level
  region: text("region"),
  expiresAt: text("expires_at").notNull(),
});

// V2G Sessions - Each session is 50 KWh
// Max 5 sessions per individual user
// Max 12 sessions per car maker
export const v2gSessions = pgTable("v2g_sessions", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  userId: varchar("user_id").notNull(), // References users.id
  vehicleId: varchar("vehicle_id").notNull(), // References vehicles.id
  manufacturer: text("manufacturer").notNull(), // Car maker for tracking limits
  kwh: real("kwh").notNull().default(50), // Each session is 50 KWh
  startTime: text("start_time").notNull(),
  endTime: text("end_time"),
  status: text("status").notNull().default('active'), // active, completed, cancelled
  gridRegion: text("grid_region"),
  incentiveEarned: real("incentive_earned").default(0),
});

export const insertGridInvitationSchema = createInsertSchema(gridInvitations).omit({
  id: true,
});

export const insertV2GSessionSchema = createInsertSchema(v2gSessions).omit({
  id: true,
});

// Chargers - Charging station infrastructure
export const chargers = pgTable("chargers", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  siteId: varchar("site_id"), // References sites.id
  model: text("model").notNull(), // Charger model
  firmware: text("firmware"), // Firmware version
  power: integer("power").notNull(), // Power capacity in kW
  status: text("status").notNull().default('active'), // active, maintenance, offline
  lastHeartbeat: text("last_heartbeat"), // Last communication timestamp
  v2gEnabled: boolean("v2g_enabled").default(false),
  connectors: integer("connectors").default(1), // Number of connectors
  lat: real("lat"), // Location latitude
  lng: real("lng"), // Location longitude
});

// Sites - Grid site/location management
export const sites = pgTable("sites", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  name: text("name").notNull(),
  capacity: integer("capacity").notNull(), // Total capacity in kW
  currentLoad: integer("current_load").default(0), // Current load in kW
  chargerCount: integer("charger_count").default(0),
  tariffId: varchar("tariff_id"), // References tariffs.id
  lat: real("lat"), // Location latitude
  lng: real("lng"), // Location longitude
  address: text("address"),
  status: text("status").default('operational'), // operational, maintenance, offline
});

// Tariffs - Pricing and billing configuration
export const tariffs = pgTable("tariffs", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  name: text("name").notNull(),
  chargingRate: real("charging_rate").notNull(), // $/kWh for charging
  v2gCreditRate: real("v2g_credit_rate").notNull(), // $/kWh credit for V2G
  scope: text("scope"), // Global, or specific sites
  effectiveDate: text("effective_date").notNull(),
  active: boolean("active").default(true),
});

// Incentive Programs
export const incentivePrograms = pgTable("incentive_programs", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  name: text("name").notNull(),
  type: text("type").notNull(), // welcome, threshold, referral, challenge
  status: text("status").default('active'), // active, scheduled, ended
  enrollments: integer("enrollments").default(0),
  conversions: integer("conversions").default(0),
  creditsIssued: integer("credits_issued").default(0),
  startDate: text("start_date"),
  endDate: text("end_date"),
  rules: text("rules"), // JSON string of program rules
});

// System Alerts
export const alerts = pgTable("alerts", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  assetId: varchar("asset_id"), // ID of related asset (charger, site, vehicle)
  assetType: text("asset_type"), // charger, site, vehicle, system
  type: text("type").notNull(), // Connectivity, Temperature, Capacity, etc.
  severity: text("severity").notNull(), // critical, warning, info
  message: text("message").notNull(),
  triggered: text("triggered").notNull(), // Timestamp when triggered
  state: text("state").default('active'), // active, acknowledged, resolved
  playbook: text("playbook"), // Associated playbook/remediation steps
});

// Hot Zones - Convert Zod schema to database table
export const hotZones = pgTable("hot_zones", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  name: text("name").notNull(),
  northBound: real("north_bound").notNull(),
  southBound: real("south_bound").notNull(),
  eastBound: real("east_bound").notNull(),
  westBound: real("west_bound").notNull(),
  stressLevel: integer("stress_level").notNull(), // 1-10 grid stress intensity
  active: boolean("active").default(true),
  priority: integer("priority").notNull(), // affects EV scoring in this zone
});

export const insertChargerSchema = createInsertSchema(chargers).omit({ id: true });
export const insertSiteSchema = createInsertSchema(sites).omit({ id: true });
export const insertTariffSchema = createInsertSchema(tariffs).omit({ id: true });
export const insertIncentiveProgramSchema = createInsertSchema(incentivePrograms).omit({ id: true });
export const insertAlertSchema = createInsertSchema(alerts).omit({ id: true });
export const insertHotZoneSchema = createInsertSchema(hotZones).omit({ id: true });

export type Charger = typeof chargers.$inferSelect;
export type InsertCharger = z.infer<typeof insertChargerSchema>;
export type Site = typeof sites.$inferSelect;
export type InsertSite = z.infer<typeof insertSiteSchema>;
export type Tariff = typeof tariffs.$inferSelect;
export type InsertTariff = z.infer<typeof insertTariffSchema>;
export type IncentiveProgram = typeof incentivePrograms.$inferSelect;
export type InsertIncentiveProgram = z.infer<typeof insertIncentiveProgramSchema>;
export type Alert = typeof alerts.$inferSelect;
export type InsertAlert = z.infer<typeof insertAlertSchema>;
export type HotZoneDB = typeof hotZones.$inferSelect;
export type InsertHotZoneDB = z.infer<typeof insertHotZoneSchema>;

// Regional statistics type (computed, not stored)
export const regionStatSchema = z.object({
  regionId: z.string(),
  bounds: z.object({
    north: z.number(),
    south: z.number(),
    east: z.number(),
    west: z.number(),
  }),
  vehicleCount: z.number(),
  chargingCount: z.number(),
  dischargingCount: z.number(),
  idleCount: z.number(),
  offlineCount: z.number(),
  totalCapacity: z.number(),
  averageSOC: z.number(),
});

export type InsertUser = z.infer<typeof insertUserSchema>;
export type User = typeof users.$inferSelect;
export type InsertVehicle = z.infer<typeof insertVehicleSchema>;
export type Vehicle = typeof vehicles.$inferSelect;
export type InsertV2GCall = z.infer<typeof insertV2GCallSchema>;
export type V2GCall = typeof v2gCalls.$inferSelect;
export type InsertV2GSession = z.infer<typeof insertV2GSessionSchema>;
export type V2GSession = typeof v2gSessions.$inferSelect;
export type InsertGridInvitation = z.infer<typeof insertGridInvitationSchema>;
export type GridInvitation = typeof gridInvitations.$inferSelect;
export type RegionStat = z.infer<typeof regionStatSchema>;

// Enhanced EV scoring and dispatch features
export const eligibilityFactorSchema = z.object({
  availability: z.number(), // plugged + idle duration score
  socBuffer: z.number(), // SoC buffer beyond user minimum
  consent: z.boolean(), // user consent for V2G participation
  batteryHealth: z.number(), // battery health % (100 = perfect)
  batteryFatigue: z.number(), // fatigue penalty based on weekly cycles
  incentiveClass: z.string(), // tariff/incentive category
  technicalReadiness: z.boolean(), // bidirectional capability
  locationPriority: z.number(), // hot zone bonus
  fleetBonus: z.number(), // fleet vs private bias
});

export const evScoringSchema = z.object({
  vehicleId: z.string(),
  eligibilityScore: z.number(), // 0-100 computed score
  factors: eligibilityFactorSchema,
  surplusKWh: z.number(), // available energy above user minimum
  maxDischargeKW: z.number(), // max power output capability
  lastRotation: z.string(), // timestamp of last dispatch
  weeklyDischarges: z.number(), // fairness tracking
});

export const hotZoneSchema = z.object({
  id: z.string(),
  name: z.string(),
  bounds: z.object({
    north: z.number(),
    south: z.number(),
    east: z.number(),
    west: z.number(),
  }),
  stressLevel: z.number(), // 1-10 grid stress intensity
  active: z.boolean(), // operator can toggle on/off
  priority: z.number(), // affects EV scoring in this zone
});

export const dispatchPlanSchema = z.object({
  id: z.string(),
  targetKW: z.number(), // operator's target power requirement
  selectedVehicles: z.array(z.string()), // vehicle IDs
  totalAvailableKW: z.number(), // sum of selected vehicles' max power
  shortfall: z.number(), // negative if target exceeded, positive if unmet
  estimatedDuration: z.number(), // minutes
  created: z.string(), // timestamp
});

export const gridEventSchema = z.object({
  id: z.string(),
  type: z.enum(['frequency_regulation', 'demand_response', 'peak_shaving', 'grid_stress']),
  magnitude: z.number(), // MW required
  location: z.object({ lat: z.number(), lng: z.number() }).optional(),
  region: z.string().optional(),
  priority: z.number(), // 1-10
  timestamp: z.string(),
  expiresAt: z.string(),
  status: z.enum(['active', 'resolved', 'expired']),
});

export const economicsDataSchema = z.object({
  vehicleId: z.string(),
  annualRevenue: z.number(), // $700-$16,000 range
  monthlyEarnings: z.number(),
  billSavings: z.number(), // TOU arbitrage savings
  incentivesReceived: z.number(), // rebates, subsidies
  co2Avoided: z.number(), // kg CO2 equivalent
  batteryDegradation: z.number(), // % health impact
  totalDischarges: z.number(), // lifetime count
});

export const weightConfigSchema = z.object({
  availability: z.number(), // 0-1 weight
  socBuffer: z.number(),
  batteryHealth: z.number(),
  locationPriority: z.number(),
  fleetBias: z.number(),
  fairnessRotation: z.number(),
  technicalReadiness: z.number(),
  consentRequired: z.number(),
});

export type EligibilityFactor = z.infer<typeof eligibilityFactorSchema>;
export type EVScoring = z.infer<typeof evScoringSchema>;
export type HotZone = z.infer<typeof hotZoneSchema>;
export type DispatchPlan = z.infer<typeof dispatchPlanSchema>;
export type GridEvent = z.infer<typeof gridEventSchema>;
export type EconomicsData = z.infer<typeof economicsDataSchema>;
export type WeightConfig = z.infer<typeof weightConfigSchema>;

// Notification subscription for push notifications
export const notificationSubscriptions = pgTable("notification_subscriptions", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  userId: text("user_id").notNull(),
  endpoint: text("endpoint").notNull(),
  p256dh: text("p256dh").notNull(),
  auth: text("auth").notNull(),
  platform: text("platform").notNull(),
  createdAt: text("created_at").notNull().default(sql`now()`)
});

export const insertNotificationSubscriptionSchema = createInsertSchema(notificationSubscriptions).omit({ 
  id: true, 
  createdAt: true 
});
export type InsertNotificationSubscription = z.infer<typeof insertNotificationSubscriptionSchema>;
export type SelectNotificationSubscription = typeof notificationSubscriptions.$inferSelect;

// Ping notifications for mobile users
export const pings = pgTable("pings", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  type: text("type").notNull(),
  title: text("title").notNull(),
  message: text("message").notNull(),
  payload: text("payload"),
  priority: text("priority").notNull().default('normal'),
  ttl: integer("ttl").notNull().default(3600),
  sentAt: text("sent_at").notNull().default(sql`now()`),
  userId: text("user_id"),
  region: text("region"),
  vehicleId: text("vehicle_id")
});

export const insertPingSchema = createInsertSchema(pings).omit({ 
  id: true, 
  sentAt: true 
});
export type InsertPing = z.infer<typeof insertPingSchema>;
export type SelectPing = typeof pings.$inferSelect;
