import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { StatusBadge, type ChargerStatus } from "./StatusBadge";
import { Play, Settings, MapPin } from "lucide-react";
import { cn } from "@/lib/utils";

interface ChargerCardProps {
  id: string;
  name: string;
  location: string;
  status: ChargerStatus;
  lastActivity?: string;
  power?: string;
  onStart?: () => void;
  onSettings?: () => void;
  onClick?: () => void;
}

export function ChargerCard({
  id,
  name,
  location,
  status,
  lastActivity,
  power,
  onStart,
  onSettings,
  onClick,
}: ChargerCardProps) {
  return (
    <Card
      className={cn(
        "hover-elevate cursor-pointer transition-all",
        onClick && "cursor-pointer"
      )}
      onClick={onClick}
      data-testid={`card-charger-${id}`}
    >
      <CardHeader className="pb-3">
        <div className="flex items-start justify-between gap-2">
          <div className="flex-1 min-w-0">
            <CardTitle className="text-lg truncate">{name}</CardTitle>
            <div className="flex items-center gap-1 text-sm text-muted-foreground mt-1">
              <MapPin className="h-3 w-3" />
              <span className="truncate">{location}</span>
            </div>
          </div>
          <StatusBadge status={status} />
        </div>
      </CardHeader>
      <CardContent className="space-y-3">
        {power && (
          <div className={cn(
            "text-2xl font-bold",
            status === "charging" ? "text-primary" : "text-muted-foreground"
          )}>
            {power}
            <span className="text-sm text-muted-foreground ml-1">kW</span>
          </div>
        )}
        {lastActivity && (
          <div className="text-xs text-muted-foreground">
            Last activity: {lastActivity}
          </div>
        )}
        <div className="flex gap-2">
          <Button
            variant="outline"
            size="sm"
            className="flex-1"
            onClick={(e) => {
              e.stopPropagation();
              onStart?.();
            }}
            disabled={status === "offline" || status === "charging"}
            data-testid={`button-start-${id}`}
          >
            <Play className="h-4 w-4 mr-1" />
            Start
          </Button>
          <Button
            variant="ghost"
            size="icon"
            className="h-8 w-8"
            onClick={(e) => {
              e.stopPropagation();
              onSettings?.();
            }}
            data-testid={`button-settings-${id}`}
          >
            <Settings className="h-4 w-4" />
          </Button>
        </div>
      </CardContent>
    </Card>
  );
}
