import { useState } from "react";
import { Button } from "@/components/ui/button";
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogDescription,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
} from "@/components/ui/alert-dialog";
import { cn } from "@/lib/utils";
import { Zap, Leaf, Gauge } from "lucide-react";

export type ChargingMode = "standard" | "eco" | "boost";

interface ChargingModeSelectorProps {
  value: ChargingMode;
  onChange: (mode: ChargingMode) => void;
  disabled?: boolean;
}

const modes = [
  {
    id: "eco" as const,
    label: "Eco",
    icon: Leaf,
    description: "Reduced current for lower demand",
  },
  {
    id: "standard" as const,
    label: "Standard",
    icon: Gauge,
    description: "Balanced everyday charging",
  },
  {
    id: "boost" as const,
    label: "Boost",
    icon: Zap,
    description: "Maximum power for urgent needs",
  },
];

export function ChargingModeSelector({
  value,
  onChange,
  disabled,
}: ChargingModeSelectorProps) {
  const [showBoostWarning, setShowBoostWarning] = useState(false);

  const handleModeClick = (mode: ChargingMode) => {
    if (mode === "boost") {
      setShowBoostWarning(true);
    } else {
      onChange(mode);
    }
  };

  const confirmBoost = () => {
    onChange("boost");
    setShowBoostWarning(false);
  };

  return (
    <>
      <div className="flex gap-2" data-testid="selector-charging-mode">
        {modes.map((mode) => {
          const Icon = mode.icon;
          const isActive = value === mode.id;

          return (
            <Button
              key={mode.id}
              variant={isActive ? "default" : "outline"}
              className={cn(
                "flex-1 flex-col h-auto py-3 gap-1",
                !isActive && "bg-card"
              )}
              onClick={() => handleModeClick(mode.id)}
              disabled={disabled}
              data-testid={`button-mode-${mode.id}`}
            >
              <Icon className="h-5 w-5" />
              <div className="font-medium">{mode.label}</div>
              <div className="text-xs opacity-80 font-normal">
                {mode.description}
              </div>
            </Button>
          );
        })}
      </div>

      <AlertDialog open={showBoostWarning} onOpenChange={setShowBoostWarning}>
        <AlertDialogContent>
          <AlertDialogHeader>
            <AlertDialogTitle>Enable Boost Mode?</AlertDialogTitle>
            <AlertDialogDescription>
              Boost mode will charge at maximum available power. This may
              significantly increase peak demand and could impact your
              building's capacity limits.
            </AlertDialogDescription>
          </AlertDialogHeader>
          <AlertDialogFooter>
            <AlertDialogCancel data-testid="button-cancel-boost">Cancel</AlertDialogCancel>
            <AlertDialogAction onClick={confirmBoost} data-testid="button-confirm-boost">
              Enable Boost
            </AlertDialogAction>
          </AlertDialogFooter>
        </AlertDialogContent>
      </AlertDialog>
    </>
  );
}
