import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { singaporeChargers } from "@shared/singapore-data";
import {
  Battery,
  DollarSign,
  TrendingUp,
  MapPin,
  Clock,
  Calendar,
  CreditCard,
  ArrowUpCircle,
  Settings,
  Gift,
  Award,
  Wallet,
  Activity,
} from "lucide-react";

function getCurrentPrice(charger: any) {
  const hour = new Date().getHours();
  const isPeakHour = hour >= 18 && hour < 22;
  return isPeakHour ? charger.peakPricePerKwh : charger.pricePerKwh;
}

function getStatusColor(status: string) {
  switch (status) {
    case "idle":
      return "border-green-500 text-green-600";
    case "charging":
      return "border-blue-500 text-blue-600";
    case "offline":
      return "border-red-500 text-red-600";
    default:
      return "";
  }
}

export default function Aggregator() {
  const hour = new Date().getHours();
  const isPeakHour = hour >= 18 && hour < 22;

  // Group chargers by area for pricing benchmarks
  const pricingByArea = singaporeChargers.reduce((acc: any, charger) => {
    const area = charger.area;
    if (!acc[area]) {
      acc[area] = [];
    }
    acc[area].push(getCurrentPrice(charger));
    return acc;
  }, {});

  const areaPricing = Object.entries(pricingByArea).map(([area, prices]: [string, any]) => ({
    area,
    minPrice: Math.min(...prices),
    maxPrice: Math.max(...prices),
    avgPrice: (prices.reduce((a: number, b: number) => a + b, 0) / prices.length),
  })).sort((a, b) => a.avgPrice - b.avgPrice);

  return (
    <div className="space-y-8">
      <div>
        <h1 className="text-4xl font-bold">Aggregator Network</h1>
        <p className="text-muted-foreground mt-2">
          Real-time pricing and V2G participation across Singapore
        </p>
      </div>

      {/* Singapore Pricing Benchmarks */}
      <div>
        <h2 className="text-2xl font-semibold mb-4 flex items-center gap-2">
          <DollarSign className="h-6 w-6 text-primary" />
          Singapore Pricing Benchmarks
        </h2>
        <div className="grid grid-cols-1 md:grid-cols-2 xl:grid-cols-3 gap-4">
          {areaPricing.slice(0, 6).map((pricing) => (
            <Card key={pricing.area} data-testid={`card-pricing-${pricing.area.toLowerCase().replace(/\s+/g, '-')}`}>
              <CardHeader className="pb-2">
                <CardTitle className="text-sm font-medium text-muted-foreground flex items-center gap-2">
                  <MapPin className="h-4 w-4" />
                  {pricing.area}
                </CardTitle>
              </CardHeader>
              <CardContent className="space-y-2">
                <div className="flex items-baseline gap-2">
                  <div className="text-2xl font-bold text-primary">
                    ${pricing.avgPrice.toFixed(2)}
                  </div>
                  <div className="text-xs text-muted-foreground">avg/kWh</div>
                </div>
                <div className="flex items-center justify-between text-xs">
                  <span className="text-muted-foreground">Range:</span>
                  <span className="font-semibold">
                    ${pricing.minPrice.toFixed(2)} - ${pricing.maxPrice.toFixed(2)}
                  </span>
                </div>
              </CardContent>
            </Card>
          ))}
        </div>
      </div>

      {/* Pricing Tiers */}
      <div>
        <h2 className="text-2xl font-semibold mb-4 flex items-center gap-2">
          <CreditCard className="h-6 w-6 text-primary" />
          Pricing Tiers
        </h2>
        <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
          <Card data-testid="card-pricing-per-discharge">
            <CardHeader>
              <CardTitle className="text-base flex items-center gap-2">
                <Battery className="h-5 w-5 text-primary" />
                Per Discharge
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-3">
              <div className="text-3xl font-bold text-primary">$0.72</div>
              <div className="text-sm text-muted-foreground">per kWh discharged</div>
              <div className="pt-3 border-t space-y-2">
                <div className="flex items-center justify-between text-sm">
                  <span className="text-muted-foreground">Peak Rate</span>
                  <span className="font-semibold">$0.85/kWh</span>
                </div>
                <div className="flex items-center justify-between text-sm">
                  <span className="text-muted-foreground">Off-Peak Rate</span>
                  <span className="font-semibold">$0.65/kWh</span>
                </div>
              </div>
              <Badge variant="outline" className="w-full justify-center">
                Best for flexible usage
              </Badge>
            </CardContent>
          </Card>

          <Card data-testid="card-pricing-flat-fee">
            <CardHeader>
              <CardTitle className="text-base flex items-center gap-2">
                <DollarSign className="h-5 w-5 text-primary" />
                Flat Fee
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-3">
              <div className="text-3xl font-bold text-primary">$50-$200</div>
              <div className="text-sm text-muted-foreground">per month</div>
              <div className="pt-3 border-t space-y-2">
                <div className="flex items-center justify-between text-sm">
                  <span className="text-muted-foreground">Basic Plan</span>
                  <span className="font-semibold">$50/mo</span>
                </div>
                <div className="flex items-center justify-between text-sm">
                  <span className="text-muted-foreground">Premium Plan</span>
                  <span className="font-semibold">$120/mo</span>
                </div>
                <div className="flex items-center justify-between text-sm">
                  <span className="text-muted-foreground">Enterprise Plan</span>
                  <span className="font-semibold">$200/mo</span>
                </div>
              </div>
              <Badge variant="outline" className="w-full justify-center">
                Unlimited usage
              </Badge>
            </CardContent>
          </Card>

          <Card data-testid="card-pricing-time-based">
            <CardHeader>
              <CardTitle className="text-base flex items-center gap-2">
                <Clock className="h-5 w-5 text-primary" />
                Time-Based
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-3">
              <div className="text-3xl font-bold text-primary">$1.20</div>
              <div className="text-sm text-muted-foreground">per hour</div>
              <div className="pt-3 border-t space-y-2">
                <div className="flex items-center justify-between text-sm">
                  <span className="text-muted-foreground">Daytime</span>
                  <span className="font-semibold">$1.50/hr</span>
                </div>
                <div className="flex items-center justify-between text-sm">
                  <span className="text-muted-foreground">Evening</span>
                  <span className="font-semibold">$1.20/hr</span>
                </div>
                <div className="flex items-center justify-between text-sm">
                  <span className="text-muted-foreground">Night</span>
                  <span className="font-semibold">$0.80/hr</span>
                </div>
              </div>
              <Badge variant="outline" className="w-full justify-center">
                Pay for charging time
              </Badge>
            </CardContent>
          </Card>
        </div>
      </div>

      {/* Singapore-Wide Active Incentives */}
      <div>
        <h2 className="text-2xl font-semibold mb-4 flex items-center gap-2">
          <Activity className="h-6 w-6 text-primary" />
          Active Incentives (Singapore-Wide)
        </h2>
        <div className="grid grid-cols-1 md:grid-cols-2 xl:grid-cols-3 gap-4">
          <Card data-testid="card-incentive-off-peak">
            <CardContent className="pt-6">
              <div className="p-3 bg-primary/10 border border-primary/20 rounded-lg">
                <div className="font-medium text-sm mb-1">Off-Peak Bonus</div>
                <div className="text-xs text-muted-foreground mb-2">Charge between 11PM-7AM</div>
                <Badge variant="default" className="text-xs">+$0.05/kWh</Badge>
              </div>
            </CardContent>
          </Card>
          <Card data-testid="card-incentive-v2g">
            <CardContent className="pt-6">
              <div className="p-3 bg-primary/10 border border-primary/20 rounded-lg">
                <div className="font-medium text-sm mb-1">V2G Participation</div>
                <div className="text-xs text-muted-foreground mb-2">Active grid support</div>
                <Badge variant="default" className="text-xs">+10% Bonus</Badge>
              </div>
            </CardContent>
          </Card>
          <Card data-testid="card-incentive-green">
            <CardContent className="pt-6">
              <div className="p-3 bg-primary/10 border border-primary/20 rounded-lg">
                <div className="font-medium text-sm mb-1">Green Energy Charging</div>
                <div className="text-xs text-muted-foreground mb-2">Use renewable sources</div>
                <Badge variant="default" className="text-xs">+$0.03/kWh</Badge>
              </div>
            </CardContent>
          </Card>
        </div>
      </div>

      {/* EV Owner Incentive Dashboard */}
      <div>
        <h2 className="text-2xl font-semibold mb-4 flex items-center gap-2">
          <Gift className="h-6 w-6 text-primary" />
          My Incentive Dashboard
        </h2>
        <div className="grid grid-cols-1 md:grid-cols-2 xl:grid-cols-3 gap-4">
          <Card data-testid="card-incentive-wallet" className="md:col-span-2">
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Wallet className="h-5 w-5 text-primary" />
                My Incentive Wallet
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="flex items-center justify-between p-4 bg-primary/10 border border-primary/20 rounded-lg">
                <div>
                  <div className="text-sm text-muted-foreground mb-1">Available Credits</div>
                  <div className="text-4xl font-bold" data-testid="text-wallet-balance">
                    $24.50
                  </div>
                </div>
                <CreditCard className="h-12 w-12 text-primary" />
              </div>
              <div className="grid grid-cols-2 gap-3">
                <div className="p-3 bg-muted/50 rounded-lg">
                  <div className="text-xs text-muted-foreground mb-1">Earned This Month</div>
                  <div className="text-xl font-bold" data-testid="text-earned-month">$18.75</div>
                </div>
                <div className="p-3 bg-muted/50 rounded-lg">
                  <div className="text-xs text-muted-foreground mb-1">Redeemed</div>
                  <div className="text-xl font-bold" data-testid="text-redeemed">$142.30</div>
                </div>
              </div>
              <Button className="w-full" data-testid="button-redeem-credits">
                <Award className="h-4 w-4 mr-2" />
                Redeem Credits
              </Button>
            </CardContent>
          </Card>
          <Card data-testid="card-my-active-incentives">
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Activity className="h-5 w-5 text-primary" />
                My Active Incentives
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-3">
              <div className="p-3 bg-primary/10 border border-primary/20 rounded-lg">
                <div className="font-medium text-sm mb-1">Off-Peak Bonus</div>
                <div className="text-xs text-muted-foreground mb-2">Active</div>
                <Badge variant="default" className="text-xs">+$0.05/kWh</Badge>
              </div>
              <div className="p-3 bg-primary/10 border border-primary/20 rounded-lg">
                <div className="font-medium text-sm mb-1">V2G Participation</div>
                <div className="text-xs text-muted-foreground mb-2">Enrolled</div>
                <Badge variant="default" className="text-xs">+10% Bonus</Badge>
              </div>
              <Button variant="outline" className="w-full" size="sm" data-testid="button-view-incentives">
                View All Incentives
              </Button>
            </CardContent>
          </Card>
        </div>
      </div>

      {/* V2G Status */}
      <div>
        <h2 className="text-2xl font-semibold mb-4 flex items-center gap-2">
          <Battery className="h-6 w-6 text-primary" />
          V2G Status
        </h2>
        <Card data-testid="card-v2g-status">
          <CardContent className="pt-6">
            <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
              <div className="flex items-center justify-between p-3 bg-primary/10 border border-primary/20 rounded-lg">
                <div>
                  <div className="font-medium text-sm">Vehicle SOC</div>
                  <div className="text-xs text-muted-foreground">Battery Level</div>
                </div>
                <div className="text-right">
                  <div className="text-2xl font-bold text-primary" data-testid="text-soc">78%</div>
                  <div className="text-xs text-muted-foreground">58 kWh</div>
                </div>
              </div>
              <div className="p-3 bg-muted/50 rounded-lg">
                <div className="flex items-center justify-between text-sm mb-2">
                  <span className="text-muted-foreground">Min SOC Limit</span>
                  <span className="font-semibold" data-testid="text-min-soc">50%</span>
                </div>
                <div className="flex items-center justify-between text-sm">
                  <span className="text-muted-foreground">Available to Grid</span>
                  <span className="font-semibold text-primary" data-testid="text-available-grid">21 kWh</span>
                </div>
              </div>
              <div className="flex gap-2">
                <Button variant="outline" className="flex-1" size="sm" data-testid="button-discharge">
                  <ArrowUpCircle className="h-4 w-4 mr-2" />
                  Discharge
                </Button>
                <Button variant="outline" size="sm" data-testid="button-v2g-settings">
                  <Settings className="h-4 w-4" />
                </Button>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>

      {/* V2G Earnings & Scheduling Dashboard */}
      <div>
        <h2 className="text-2xl font-semibold mb-4 flex items-center gap-2">
          <TrendingUp className="h-6 w-6 text-primary" />
          V2G Earnings & Scheduling
        </h2>
        <div className="grid grid-cols-1 md:grid-cols-2 xl:grid-cols-3 gap-6">
          <Card data-testid="card-v2g-earnings">
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <DollarSign className="h-5 w-5 text-primary" />
                Earnings Summary
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="grid grid-cols-2 gap-3">
                <div className="p-3 bg-primary/10 border border-primary/20 rounded-lg">
                  <div className="text-xs text-muted-foreground mb-1">Today</div>
                  <div className="text-2xl font-bold text-primary" data-testid="text-earnings-today">$3.24</div>
                </div>
                <div className="p-3 bg-muted/50 rounded-lg">
                  <div className="text-xs text-muted-foreground mb-1">This Week</div>
                  <div className="text-2xl font-bold" data-testid="text-earnings-week">$18.52</div>
                </div>
              </div>
              <div className="p-4 bg-muted/50 rounded-lg">
                <div className="text-sm text-muted-foreground mb-1">This Month</div>
                <div className="text-3xl font-bold" data-testid="text-earnings-month">$47.18</div>
              </div>
              <div className="space-y-2">
                <div className="flex items-center justify-between text-sm">
                  <span className="text-muted-foreground">Total Discharged</span>
                  <span className="font-semibold" data-testid="text-total-discharged">65.5 kWh</span>
                </div>
                <div className="flex items-center justify-between text-sm">
                  <span className="text-muted-foreground">Grid Events</span>
                  <span className="font-semibold" data-testid="text-grid-events">3 this week</span>
                </div>
              </div>
            </CardContent>
          </Card>

          <Card data-testid="card-discharge-schedule">
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Calendar className="h-5 w-5 text-primary" />
                Discharge Schedule
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-3">
              <div className="p-3 bg-primary/10 border border-primary/20 rounded-lg">
                <div className="flex items-center justify-between mb-2">
                  <div className="font-medium text-sm">Tonight 7-9PM</div>
                  <Badge variant="default" className="text-xs">Scheduled</Badge>
                </div>
                <div className="text-xs text-muted-foreground mb-1">Max 15 kWh discharge</div>
                <div className="text-sm font-semibold text-primary">Est. $16.20 earnings</div>
              </div>
              <div className="p-3 bg-muted/50 rounded-lg">
                <div className="flex items-center justify-between mb-2">
                  <div className="font-medium text-sm">Tomorrow 6-8AM</div>
                  <Badge variant="outline" className="text-xs">Pending</Badge>
                </div>
                <div className="text-xs text-muted-foreground mb-1">Auto-discharge enabled</div>
                <div className="text-sm font-semibold">Min SOC: 60%</div>
              </div>
              <Button variant="outline" className="w-full" size="sm" data-testid="button-schedule-discharge">
                <Calendar className="h-4 w-4 mr-2" />
                Add Schedule
              </Button>
            </CardContent>
          </Card>
        </div>
      </div>
    </div>
  );
}
