import { ChargerCard } from "@/components/ChargerCard";
import { ChargerMap } from "@/components/ChargerMap";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Input } from "@/components/ui/input";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { Search, Plus, Filter, Navigation, Calendar, Clock, CreditCard, Activity, MapPin, Star, ArrowUpCircle } from "lucide-react";
import { useState } from "react";
import { singaporeChargers } from "@shared/singapore-data";

const mockChargers = [
  { id: "1", name: "Orchard Central A1", location: "Orchard", status: "charging" as const, power: "7.4", lastActivity: "Active now" },
  { id: "2", name: "Orchard Central A2", location: "Orchard", status: "idle" as const, lastActivity: "2 hours ago" },
  { id: "3", name: "Marina Bay Sands B1", location: "Marina Bay", status: "charging" as const, power: "10.8", lastActivity: "Active now" },
  { id: "4", name: "Changi Business Park C1", location: "Changi", status: "idle" as const, lastActivity: "Yesterday" },
  { id: "5", name: "Toa Payoh Hub D1", location: "Toa Payoh", status: "offline" as const, lastActivity: "3 days ago" },
  { id: "6", name: "Jurong Point E1", location: "Jurong West", status: "idle" as const, lastActivity: "5 hours ago" },
  { id: "7", name: "Punggol Digital District F1", location: "Punggol", status: "idle" as const, lastActivity: "1 hour ago" },
  { id: "8", name: "Punggol Digital District F2", location: "Punggol", status: "charging" as const, power: "22.0", lastActivity: "Active now" },
  { id: "9", name: "Sentosa Gateway G1", location: "Sentosa", status: "idle" as const, lastActivity: "3 hours ago" },
  { id: "10", name: "Seletar Aerospace Park H1", location: "Seletar", status: "idle" as const, lastActivity: "6 hours ago" },
];

function getCurrentPrice(charger: any) {
  const hour = new Date().getHours();
  const isPeakHour = hour >= 18 && hour < 22;
  return isPeakHour ? charger.peakPricePerKwh : charger.pricePerKwh;
}

function getStatusColor(status: string) {
  switch (status) {
    case "idle":
      return "border-green-500 text-green-600";
    case "charging":
      return "border-blue-500 text-blue-600";
    case "offline":
      return "border-red-500 text-red-600";
    default:
      return "";
  }
}

export default function Chargers() {
  const [searchQuery, setSearchQuery] = useState("");
  const [selectedCharger, setSelectedCharger] = useState(singaporeChargers.length > 0 ? singaporeChargers[0] : undefined);
  const [favoriteChargers] = useState(["1", "3", "5"]);

  const filteredChargers = singaporeChargers.filter((charger) =>
    charger.chargerName.toLowerCase().includes(searchQuery.toLowerCase()) ||
    charger.area.toLowerCase().includes(searchQuery.toLowerCase())
  );

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-4xl font-bold">Chargers</h1>
          <p className="text-muted-foreground mt-2">
            View and manage all charging stations
          </p>
        </div>
        <Button data-testid="button-add-charger">
          <Plus className="h-4 w-4 mr-2" />
          Add Charger
        </Button>
      </div>

      <div className="flex gap-4">
        <div className="relative flex-1">
          <Search className="absolute left-3 top-1/2 -translate-y-1/2 h-4 w-4 text-muted-foreground" />
          <Input
            placeholder="Search chargers..."
            value={searchQuery}
            onChange={(e) => setSearchQuery(e.target.value)}
            className="pl-9"
            data-testid="input-search-chargers"
          />
        </div>
        <Button variant="outline" data-testid="button-filter">
          <Filter className="h-4 w-4 mr-2" />
          Filter
        </Button>
      </div>

      {singaporeChargers.length > 0 && (
        <div>
          <h2 className="text-2xl font-semibold mb-4">Singapore Charger Network Map</h2>
          <ChargerMap
            chargers={singaporeChargers}
            selectedCharger={selectedCharger}
            onChargerSelect={setSelectedCharger}
          />
        </div>
      )}

      {/* Quick Actions */}
      <div>
        <h2 className="text-2xl font-semibold mb-4">Quick Actions</h2>
        <Card data-testid="card-quick-actions">
          <CardContent className="pt-6">
            <div className="grid grid-cols-1 md:grid-cols-2 xl:grid-cols-4 gap-3">
              <Button className="w-full" data-testid="button-route-planner">
                <Navigation className="h-4 w-4 mr-2" />
                Route Planner
              </Button>
              <Button variant="outline" className="w-full" data-testid="button-reserve">
                <Calendar className="h-4 w-4 mr-2" />
                Reserve Charger
              </Button>
              <Button variant="outline" className="w-full" data-testid="button-my-sessions">
                <Clock className="h-4 w-4 mr-2" />
                My Sessions
              </Button>
              <Button variant="outline" className="w-full" data-testid="button-payment-methods">
                <CreditCard className="h-4 w-4 mr-2" />
                Payment Methods
              </Button>
            </div>
          </CardContent>
        </Card>
      </div>

      {/* Upcoming Grid Events */}
      <div>
        <h2 className="text-2xl font-semibold mb-4 flex items-center gap-2">
          <Activity className="h-6 w-6 text-primary" />
          Upcoming Grid Events
        </h2>
        <div className="grid grid-cols-1 md:grid-cols-2 xl:grid-cols-3 gap-4">
          <Card data-testid="card-grid-events">
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Activity className="h-5 w-5 text-primary" />
                High Demand Period
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-3">
              <div className="p-3 bg-primary/10 border border-primary/20 rounded-lg" data-testid="event-high-demand">
                <div className="flex items-center justify-between mb-2">
                  <div className="font-medium text-sm">Today 7PM - 9PM</div>
                  <Badge variant="default" className="text-xs">+50% Bonus</Badge>
                </div>
                <div className="text-xs text-muted-foreground">Earn $1.08/kWh for discharging</div>
              </div>
            </CardContent>
          </Card>
          <Card data-testid="card-off-peak">
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Activity className="h-5 w-5 text-primary" />
                Off-Peak Charging
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-3">
              <div className="p-3 bg-muted/50 rounded-lg" data-testid="event-off-peak">
                <div className="flex items-center justify-between mb-2">
                  <div className="font-medium text-sm">Tonight 11PM - 7AM</div>
                  <Badge variant="outline" className="text-xs">-30% Cost</Badge>
                </div>
                <div className="text-xs text-muted-foreground">Charge from $0.27/kWh</div>
              </div>
            </CardContent>
          </Card>
          <Card data-testid="card-grid-notifications">
            <CardHeader>
              <CardTitle>Event Notifications</CardTitle>
            </CardHeader>
            <CardContent>
              <Button variant="outline" className="w-full" size="sm" data-testid="button-event-notifications">
                Enable Notifications
              </Button>
            </CardContent>
          </Card>
        </div>
      </div>

      {/* Available Chargers */}
      <div className="grid grid-cols-1 xl:grid-cols-3 gap-6">
        <div className="xl:col-span-2">
          <Card data-testid="card-charger-list">
            <CardHeader>
              <div className="flex items-center justify-between gap-4">
                <CardTitle>Available Chargers</CardTitle>
                <div className="flex gap-2">
                  <div className="relative flex-1">
                    <Search className="absolute left-3 top-1/2 -translate-y-1/2 h-4 w-4 text-muted-foreground" />
                    <Input
                      placeholder="Search..."
                      value={searchQuery}
                      onChange={(e) => setSearchQuery(e.target.value)}
                      className="pl-9 w-64"
                      data-testid="input-search-available-chargers"
                    />
                  </div>
                  <Select defaultValue="price">
                    <SelectTrigger className="w-40" data-testid="select-sort">
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="price">Lowest Price</SelectItem>
                      <SelectItem value="distance">Nearest</SelectItem>
                      <SelectItem value="power">Max Power</SelectItem>
                      <SelectItem value="availability">Available First</SelectItem>
                    </SelectContent>
                  </Select>
                </div>
              </div>
            </CardHeader>
            <CardContent className="space-y-2 max-h-[600px] overflow-y-auto">
              {filteredChargers.map((charger) => {
                const price = getCurrentPrice(charger);
                const isFavorite = favoriteChargers.includes(charger.chargerId);

                return (
                  <div
                    key={charger.chargerId}
                    className="flex items-center gap-4 p-4 rounded-lg border border-border hover-elevate cursor-pointer"
                    onClick={() => setSelectedCharger(charger)}
                    data-testid={`charger-item-${charger.chargerId}`}
                  >
                    <div className="flex-1">
                      <div className="flex items-center gap-2 mb-1">
                        <h4 className="font-semibold">{charger.chargerName}</h4>
                        {isFavorite && (
                          <Star className="h-4 w-4 fill-primary text-primary" />
                        )}
                      </div>
                      <div className="flex items-center gap-1 text-sm text-muted-foreground mb-2">
                        <MapPin className="h-3 w-3" />
                        <span>{charger.area}</span>
                      </div>
                      <div className="flex items-center gap-2 text-sm flex-wrap">
                        <Badge
                          variant="outline"
                          className={getStatusColor(charger.status)}
                        >
                          {charger.status === "idle" ? "Available" : charger.status}
                        </Badge>
                        <Badge variant="outline" className="text-xs">
                          {charger.maxPower} kW
                        </Badge>
                        <Badge variant="outline" className="text-xs">
                          {charger.connectorType}
                        </Badge>
                        {charger.v2gCapable && (
                          <Badge variant="default" className="text-xs" data-testid={`v2g-badge-list-${charger.chargerId}`}>
                            <ArrowUpCircle className="h-3 w-3 mr-1" />
                            V2G
                          </Badge>
                        )}
                      </div>
                    </div>
                    <div className="text-right">
                      <div className="text-2xl font-bold text-primary">
                        ${price.toFixed(2)}
                      </div>
                      <div className="text-xs text-muted-foreground">/kWh</div>
                      {charger.status === "idle" && (
                        <Button
                          size="sm"
                          className="mt-2"
                          onClick={(e) => {
                            e.stopPropagation();
                            console.log("Start session", charger.chargerId);
                          }}
                          data-testid={`button-start-${charger.chargerId}`}
                        >
                          Start Session
                        </Button>
                      )}
                    </div>
                  </div>
                );
              })}
            </CardContent>
          </Card>
        </div>

        <div className="space-y-6">
          <Card data-testid="card-favorites">
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Star className="h-5 w-5 text-primary" />
                Favorites
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-2">
              {singaporeChargers
                .filter((c) => favoriteChargers.includes(c.chargerId))
                .map((charger) => (
                  <div
                    key={charger.chargerId}
                    className="p-3 rounded-lg bg-muted/50 cursor-pointer hover-elevate"
                    onClick={() => setSelectedCharger(charger)}
                    data-testid={`favorite-${charger.chargerId}`}
                  >
                    <div className="font-medium text-sm">{charger.chargerName}</div>
                    <div className="text-xs text-muted-foreground">{charger.area}</div>
                    <div className="flex items-center justify-between mt-2">
                      <Badge
                        variant="outline"
                        className={getStatusColor(charger.status)}
                      >
                        {charger.status === "idle" ? "Available" : charger.status}
                      </Badge>
                      <span className="text-sm font-semibold text-primary">
                        ${getCurrentPrice(charger).toFixed(2)}/kWh
                      </span>
                    </div>
                  </div>
                ))}
            </CardContent>
          </Card>
        </div>
      </div>

      <div>
        <h2 className="text-2xl font-semibold mb-4">All Chargers</h2>
        <div className="grid grid-cols-1 md:grid-cols-2 xl:grid-cols-3 gap-4">
          {mockChargers.map((charger) => (
            <ChargerCard
              key={charger.id}
              {...charger}
              onStart={() => console.log("Start", charger.id)}
              onSettings={() => console.log("Settings", charger.id)}
              onClick={() => console.log("View", charger.id)}
            />
          ))}
        </div>
      </div>
    </div>
  );
}
