import { useState } from "react";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { LiveChargingCard } from "@/components/LiveChargingCard";
import { ChargerCard } from "@/components/ChargerCard";
import {
  MapPin,
  Clock,
  Zap,
  Bell,
  Battery,
  DollarSign,
  CheckCircle,
  XCircle,
  MapPinned,
  BatteryCharging,
} from "lucide-react";
import type { ChargingMode } from "@/components/ChargingModeSelector";

export default function Dashboard() {
  const [mode, setMode] = useState<ChargingMode>("standard");
  const [gridNotifications, setGridNotifications] = useState([
    {
      id: 1,
      title: "Peak Demand Grid Call",
      location: "Marina Bay Sands",
      incentive: "$12.50",
      fastCharge: true,
      duration: "30 mins",
      timestamp: "2 mins ago",
      status: "pending"
    },
    {
      id: 2,
      title: "V2G Discharge Request",
      location: "Orchard Central",
      incentive: "$8.75",
      fastCharge: false,
      duration: "45 mins",
      timestamp: "8 mins ago",
      status: "pending"
    }
  ]);

  const handleAccept = (id: number) => {
    setGridNotifications(prev => 
      prev.map(notif => notif.id === id ? {...notif, status: "accepted"} : notif)
    );
  };

  const handleDecline = (id: number) => {
    setGridNotifications(prev => 
      prev.map(notif => notif.id === id ? {...notif, status: "declined"} : notif)
    );
  };

  return (
    <div className="space-y-6">
      <div>
        <h1 className="text-4xl font-bold">My EV Dashboard</h1>
        <p className="text-muted-foreground mt-2">
          Monitor your charging activity and available stations
        </p>
      </div>

      {/* Session Metrics */}
      <div>
        <h2 className="text-2xl font-semibold mb-4">Session Metrics</h2>
        <div className="grid grid-cols-1 md:grid-cols-2 xl:grid-cols-5 gap-4">
          <Card data-testid="card-no-sessions">
            <CardHeader className="pb-2">
              <CardTitle className="text-sm font-medium text-muted-foreground">
                No. of Sessions
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="text-3xl font-bold text-primary" data-testid="text-no-sessions">05</div>
            </CardContent>
          </Card>

          <Card data-testid="card-power-available">
            <CardHeader className="pb-2">
              <CardTitle className="text-sm font-medium text-muted-foreground">
                Power Made Available
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="text-3xl font-bold text-primary" data-testid="text-power-available">250 kW</div>
            </CardContent>
          </Card>

          <Card data-testid="card-avg-soc">
            <CardHeader className="pb-2">
              <CardTitle className="text-sm font-medium text-muted-foreground">
                Average SOC
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="text-3xl font-bold text-primary" data-testid="text-avg-soc">80%</div>
            </CardContent>
          </Card>

          <Card data-testid="card-active-sessions">
            <CardHeader className="pb-2">
              <CardTitle className="text-sm font-medium text-muted-foreground">
                Active Sessions
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="text-3xl font-bold text-primary" data-testid="text-active-sessions">01</div>
            </CardContent>
          </Card>

          <Card data-testid="card-currently-engaged">
            <CardHeader className="pb-2">
              <CardTitle className="text-sm font-medium text-muted-foreground">
                Status
              </CardTitle>
            </CardHeader>
            <CardContent>
              <Badge variant="default" className="text-sm" data-testid="badge-engaged-status">
                Currently Engaged
              </Badge>
            </CardContent>
          </Card>
        </div>
      </div>

      {/* Total Earnings */}
      <div>
        <h2 className="text-2xl font-semibold mb-4">Earnings Overview</h2>
        <Card data-testid="card-total-earnings">
          <CardContent className="pt-6">
            <div className="flex items-center justify-between p-4 bg-primary/10 border border-primary/20 rounded-lg">
              <div>
                <div className="text-sm text-muted-foreground mb-1">Total Earnings</div>
                <div className="text-5xl font-bold text-primary" data-testid="text-total-earnings">
                  $500
                </div>
                <div className="text-xs text-muted-foreground mt-1">Single user total</div>
              </div>
              <DollarSign className="h-16 w-16 text-primary" />
            </div>
          </CardContent>
        </Card>
      </div>

      {/* Grid Call Notifications */}
      <div>
        <div className="flex items-center justify-between mb-4">
          <h2 className="text-2xl font-semibold">Grid Call Notifications</h2>
          <div className="flex items-center gap-2">
            <Badge variant="outline" data-testid="badge-grid-calls-made">
              <Bell className="h-3 w-3 mr-1" />
              Grid Calls Made: 24
            </Badge>
            <Badge variant="outline" data-testid="badge-grid-calls-accepted">
              <CheckCircle className="h-3 w-3 mr-1" />
              Accepted: 5
            </Badge>
          </div>
        </div>

        <div className="grid grid-cols-1 md:grid-cols-2 gap-4 mb-4">
          <Card data-testid="card-preferred-location">
            <CardContent className="pt-6">
              <div className="flex items-center justify-between">
                <div className="flex items-center gap-2">
                  <MapPinned className="h-5 w-5 text-primary" />
                  <div>
                    <div className="text-sm text-muted-foreground">Preferred Location</div>
                    <div className="font-semibold" data-testid="text-preferred-location">Marina Bay Sands</div>
                  </div>
                </div>
              </div>
            </CardContent>
          </Card>

          <Card data-testid="card-max-discharge">
            <CardContent className="pt-6">
              <div className="flex items-center justify-between">
                <div className="flex items-center gap-2">
                  <Battery className="h-5 w-5 text-primary" />
                  <div>
                    <div className="text-sm text-muted-foreground">Max No. Discharge</div>
                    <div className="font-semibold" data-testid="text-max-discharge">12</div>
                  </div>
                </div>
              </div>
            </CardContent>
          </Card>
        </div>

        <div className="space-y-3">
          {gridNotifications.map((notification) => (
            <Card key={notification.id} data-testid={`grid-notification-${notification.id}`}>
              <CardContent className="pt-6">
                <div className="flex items-start gap-4">
                  <div className="p-3 rounded-full bg-primary/10">
                    <Bell className="h-5 w-5 text-primary" />
                  </div>
                  <div className="flex-1">
                    <div className="flex items-start justify-between gap-4 mb-3">
                      <div>
                        <h3 className="font-semibold text-lg" data-testid={`notification-title-${notification.id}`}>
                          {notification.title}
                        </h3>
                        <p className="text-sm text-muted-foreground">{notification.timestamp}</p>
                      </div>
                      {notification.status === "pending" && (
                        <Badge variant="outline">New</Badge>
                      )}
                      {notification.status === "accepted" && (
                        <Badge variant="default">Accepted</Badge>
                      )}
                      {notification.status === "declined" && (
                        <Badge variant="secondary">Declined</Badge>
                      )}
                    </div>
                    
                    <div className="grid grid-cols-2 md:grid-cols-4 gap-3 mb-4">
                      <div className="flex items-center gap-2">
                        <DollarSign className="h-4 w-4 text-muted-foreground" />
                        <div>
                          <div className="text-xs text-muted-foreground">Incentive</div>
                          <div className="font-semibold text-sm" data-testid={`notification-incentive-${notification.id}`}>
                            {notification.incentive}
                          </div>
                        </div>
                      </div>
                      
                      <div className="flex items-center gap-2">
                        <MapPin className="h-4 w-4 text-muted-foreground" />
                        <div>
                          <div className="text-xs text-muted-foreground">Location</div>
                          <div className="font-semibold text-sm" data-testid={`notification-location-${notification.id}`}>
                            {notification.location}
                          </div>
                        </div>
                      </div>
                      
                      <div className="flex items-center gap-2">
                        <BatteryCharging className="h-4 w-4 text-muted-foreground" />
                        <div>
                          <div className="text-xs text-muted-foreground">Fast Charge</div>
                          <div className="font-semibold text-sm" data-testid={`notification-fast-charge-${notification.id}`}>
                            {notification.fastCharge ? "Yes" : "No"}
                          </div>
                        </div>
                      </div>
                      
                      <div className="flex items-center gap-2">
                        <Clock className="h-4 w-4 text-muted-foreground" />
                        <div>
                          <div className="text-xs text-muted-foreground">Duration</div>
                          <div className="font-semibold text-sm" data-testid={`notification-duration-${notification.id}`}>
                            {notification.duration}
                          </div>
                        </div>
                      </div>
                    </div>

                    {notification.status === "pending" && (
                      <div className="flex gap-2">
                        <Button
                          variant="default"
                          size="sm"
                          onClick={() => handleAccept(notification.id)}
                          data-testid={`button-accept-${notification.id}`}
                        >
                          <CheckCircle className="h-4 w-4 mr-2" />
                          Accept
                        </Button>
                        <Button
                          variant="outline"
                          size="sm"
                          onClick={() => handleDecline(notification.id)}
                          data-testid={`button-decline-${notification.id}`}
                        >
                          <XCircle className="h-4 w-4 mr-2" />
                          Decline
                        </Button>
                      </div>
                    )}
                  </div>
                </div>
              </CardContent>
            </Card>
          ))}
        </div>
      </div>

      {/* Current Charging Status */}
      <div>
        <h2 className="text-2xl font-semibold mb-4">Current Charging Session</h2>
        <LiveChargingCard
          chargerName="Orchard Central A1"
          status="charging"
          currentPower={12}
          maxPower={25}
          current={32}
          energyDispensed={12.5}
          duration="30 mins"
          mode={mode}
          onModeChange={setMode}
          onStart={() => console.log("Start")}
          onStop={() => console.log("Stop")}
          onPause={() => console.log("Pause")}
          onLock={() => console.log("Lock")}
        />
      </div>

      {/* Charging History */}
      <div>
        <h2 className="text-2xl font-semibold mb-4">Charging History</h2>
        <div className="grid grid-cols-1 md:grid-cols-2 xl:grid-cols-3 gap-4">
          <Card data-testid="card-history-1">
            <CardHeader className="pb-3">
              <div className="flex items-center justify-between">
                <CardTitle className="text-lg">Orchard Central A1</CardTitle>
                <Badge variant="outline">Active</Badge>
              </div>
            </CardHeader>
            <CardContent className="space-y-3">
              <div className="flex items-center gap-2 text-sm">
                <MapPin className="h-4 w-4 text-muted-foreground" />
                <span>Orchard Road, Singapore</span>
              </div>
              <div className="flex items-center gap-2 text-sm">
                <Clock className="h-4 w-4 text-muted-foreground" />
                <span>Started: 30 mins ago</span>
              </div>
              <div className="flex items-center gap-2 text-sm">
                <Zap className="h-4 w-4 text-muted-foreground" />
                <span>Power: 50 kW</span>
              </div>
            </CardContent>
          </Card>

          <Card data-testid="card-history-2">
            <CardHeader className="pb-3">
              <div className="flex items-center justify-between">
                <CardTitle className="text-lg">Marina Bay Sands B1</CardTitle>
                <Badge variant="secondary">Completed</Badge>
              </div>
            </CardHeader>
            <CardContent className="space-y-3">
              <div className="flex items-center gap-2 text-sm">
                <MapPin className="h-4 w-4 text-muted-foreground" />
                <span>Marina Bay, Singapore</span>
              </div>
              <div className="flex items-center gap-2 text-sm">
                <Clock className="h-4 w-4 text-muted-foreground" />
                <span>Yesterday, 8:30 AM</span>
              </div>
              <div className="flex items-center gap-2 text-sm">
                <Zap className="h-4 w-4 text-muted-foreground" />
                <span>Power: 48 kW</span>
              </div>
            </CardContent>
          </Card>

          <Card data-testid="card-history-3">
            <CardHeader className="pb-3">
              <div className="flex items-center justify-between">
                <CardTitle className="text-lg">Punggol Digital District F2</CardTitle>
                <Badge variant="secondary">Completed</Badge>
              </div>
            </CardHeader>
            <CardContent className="space-y-3">
              <div className="flex items-center gap-2 text-sm">
                <MapPin className="h-4 w-4 text-muted-foreground" />
                <span>Punggol, Singapore</span>
              </div>
              <div className="flex items-center gap-2 text-sm">
                <Clock className="h-4 w-4 text-muted-foreground" />
                <span>2 days ago, 3:15 PM</span>
              </div>
              <div className="flex items-center gap-2 text-sm">
                <Zap className="h-4 w-4 text-muted-foreground" />
                <span>Power: 52 kW</span>
              </div>
            </CardContent>
          </Card>
        </div>
      </div>

      {/* All Chargers Status */}
      <div>
        <h2 className="text-2xl font-semibold mb-4">All Chargers - Live Status</h2>
        <div className="grid grid-cols-1 md:grid-cols-2 xl:grid-cols-3 gap-4">
          <ChargerCard
            id="1"
            name="Orchard Central A1"
            location="Orchard"
            status="charging"
            power="45"
            lastActivity="Active now"
            onStart={() => console.log("Start")}
            onSettings={() => console.log("Settings")}
          />
          <ChargerCard
            id="3"
            name="Marina Bay Sands B1"
            location="Marina Bay"
            status="charging"
            power="38"
            lastActivity="Active now"
            onStart={() => console.log("Start")}
            onSettings={() => console.log("Settings")}
          />
          <ChargerCard
            id="8"
            name="Punggol Digital District F2"
            location="Punggol"
            status="charging"
            power="50"
            lastActivity="Active now"
            onStart={() => console.log("Start")}
            onSettings={() => console.log("Settings")}
          />
          <ChargerCard
            id="2"
            name="Orchard Central A2"
            location="Orchard"
            status="idle"
            power="30"
            lastActivity="2 hours ago"
            onStart={() => console.log("Start")}
            onSettings={() => console.log("Settings")}
          />
          <ChargerCard
            id="4"
            name="Changi Business Park C1"
            location="Changi"
            status="idle"
            power="25"
            lastActivity="Yesterday"
            onStart={() => console.log("Start")}
            onSettings={() => console.log("Settings")}
          />
          <ChargerCard
            id="5"
            name="Toa Payoh Hub D1"
            location="Toa Payoh"
            status="offline"
            power="35"
            lastActivity="3 days ago"
            onStart={() => console.log("Start")}
            onSettings={() => console.log("Settings")}
          />
        </div>
      </div>
    </div>
  );
}
