import { useState } from "react";
import { SessionHistoryItem } from "@/components/SessionHistoryItem";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Card, CardContent } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Search, Download, Calendar, Info } from "lucide-react";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";

const mockSessions = [
  {
    sessionId: "1",
    chargerName: "Orchard Central A1",
    location: "Orchard",
    startTime: "Today 14:30",
    endTime: "16:45",
    duration: "2h 15m",
    energyDispensed: 15.3,
    mode: "standard" as const,
    outcome: "success" as const,
  },
  {
    sessionId: "2",
    chargerName: "Marina Bay Sands B1",
    location: "Marina Bay",
    startTime: "Today 08:00",
    endTime: "In progress",
    duration: "6h 45m",
    energyDispensed: 32.1,
    mode: "eco" as const,
    outcome: "success" as const,
  },
  {
    sessionId: "3",
    chargerName: "Punggol Digital District F2",
    location: "Punggol",
    startTime: "Yesterday 18:00",
    endTime: "23:30",
    duration: "5h 30m",
    energyDispensed: 28.7,
    mode: "standard" as const,
    outcome: "success" as const,
  },
  {
    sessionId: "4",
    chargerName: "Changi Business Park C1",
    location: "Changi",
    startTime: "Dec 28 20:00",
    endTime: "Interrupted",
    duration: "0h 45m",
    energyDispensed: 3.2,
    mode: "boost" as const,
    outcome: "error" as const,
  },
  {
    sessionId: "5",
    chargerName: "Marina Bay Sands B1",
    location: "Marina Bay",
    startTime: "Dec 27 09:00",
    endTime: "17:30",
    duration: "8h 30m",
    energyDispensed: 42.7,
    mode: "eco" as const,
    outcome: "success" as const,
  },
];

export default function History() {
  const [searchQuery, setSearchQuery] = useState("");

  // Session limits constants
  const MAX_SESSIONS_DISPLAY = 5;
  const MAX_SESSIONS_PER_CAR = 12;
  const displayedSessions = mockSessions.slice(0, MAX_SESSIONS_DISPLAY);

  return (
    <div className="space-y-6">
      <div>
        <h1 className="text-4xl font-bold">Charging History</h1>
        <p className="text-muted-foreground mt-2">
          View and analyze past charging sessions
        </p>
      </div>

      {/* Session Limits Info */}
      <Card className="border-primary/50 bg-primary/5" data-testid="card-session-limits">
        <CardContent className="pt-4 pb-4">
          <div className="flex items-start gap-3">
            <div className="rounded-full bg-primary/20 p-1.5">
              <Info className="h-4 w-4 text-primary" />
            </div>
            <div className="flex-1">
              <div className="flex items-center gap-4 flex-wrap">
                <div className="flex items-center gap-2">
                  <span className="text-sm text-muted-foreground">Displaying:</span>
                  <Badge variant="outline" className="bg-background">
                    {displayedSessions.length} of {mockSessions.length} sessions
                  </Badge>
                </div>
                <div className="flex items-center gap-2">
                  <span className="text-sm text-muted-foreground">Max per car:</span>
                  <Badge variant="outline" className="bg-background" data-testid="badge-max-sessions-per-car">
                    {MAX_SESSIONS_PER_CAR} sessions
                  </Badge>
                </div>
                <div className="flex items-center gap-2">
                  <span className="text-sm text-muted-foreground">Max displayed:</span>
                  <Badge variant="outline" className="bg-background" data-testid="badge-max-sessions-display">
                    {MAX_SESSIONS_DISPLAY} sessions
                  </Badge>
                </div>
              </div>
            </div>
          </div>
        </CardContent>
      </Card>

      <div className="flex flex-col sm:flex-row gap-4">
        <div className="relative flex-1">
          <Search className="absolute left-3 top-1/2 -translate-y-1/2 h-4 w-4 text-muted-foreground" />
          <Input
            placeholder="Search sessions..."
            value={searchQuery}
            onChange={(e) => setSearchQuery(e.target.value)}
            className="pl-9"
            data-testid="input-search-sessions"
          />
        </div>
        <Select defaultValue="all">
          <SelectTrigger className="w-full sm:w-48" data-testid="select-filter-charger">
            <SelectValue placeholder="All Chargers" />
          </SelectTrigger>
          <SelectContent>
            <SelectItem value="all">All Chargers</SelectItem>
            <SelectItem value="orchard">Orchard Central</SelectItem>
            <SelectItem value="marina">Marina Bay Sands</SelectItem>
            <SelectItem value="changi">Changi Business Park</SelectItem>
            <SelectItem value="punggol">Punggol Digital District</SelectItem>
          </SelectContent>
        </Select>
        <Select defaultValue="all-time">
          <SelectTrigger className="w-full sm:w-48" data-testid="select-filter-time">
            <SelectValue placeholder="All Time" />
          </SelectTrigger>
          <SelectContent>
            <SelectItem value="all-time">All Time</SelectItem>
            <SelectItem value="today">Today</SelectItem>
            <SelectItem value="week">This Week</SelectItem>
            <SelectItem value="month">This Month</SelectItem>
          </SelectContent>
        </Select>
        <Button variant="outline" data-testid="button-export">
          <Download className="h-4 w-4 mr-2" />
          Export
        </Button>
      </div>

      <div className="space-y-3">
        {displayedSessions.map((session) => (
          <SessionHistoryItem
            key={session.sessionId}
            {...session}
            onClick={() => console.log("View session", session.sessionId)}
          />
        ))}
      </div>
    </div>
  );
}
